<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Shop\Model;

use CpCreation\VitiCore\Behavior\HasTimestamp;
use CpCreation\VitiCore\Behavior\Impl\Timestamp;
use CpCreation\VitiCore\Product\Model\ProductPriceGroup;
use CpCreation\VitiCore\Product\Model\ProductType;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Ramsey\Uuid\Uuid;
use Ramsey\Uuid\UuidInterface;
use Symfony\Component\Validator\Constraints as Assert;
use Symfony\Component\Validator\Context\ExecutionContextInterface;

/**
 * Class ShippingPrice
 *
 * @ORM\Entity(repositoryClass="CpCreation\VitiCore\Shop\Repository\ShippingPriceRepository")
 * @ORM\Table(name="shop_shipping_price")
 * @ORM\HasLifecycleCallbacks()
 */
class ShippingPrice implements HasTimestamp
{
    use Timestamp;

    /**
     * @ORM\Id()
     * @ORM\Column(type="uuid")
     * @ORM\GeneratedValue(strategy="NONE")
     *
     * @var UuidInterface
     */
    private $id;

    /**
     * @ORM\Column(type="integer")
     * @Assert\NotBlank()
     *
     * @var integer
     */
    private $qtyOf;

    /**
     * @ORM\Column(type="integer")
     * @Assert\NotBlank()
     *
     * @var integer
     */
    private $qtyAt;

    /**
     * @ORM\Column(type="integer", nullable=true)
     *
     * @var int
     */
    private $packagePrice;

    /**
     * @ORM\Column(type="integer", nullable=true)
     *
     * @var int
     */
    private $bottlePrice;

    /**
     * @ORM\Column(type="boolean", options={"default" : false})
     *
     * @var bool
     */
    private $configGlobal;

    /**
     * @ORM\Column(type="decimal", precision=5, scale=4, nullable=true)
     *
     * @var float
     */
    private $privateMargePercent;

    /**
     * @ORM\Column(type="integer", nullable=true)
     *
     * @var int
     */
    private $privateMarge;

    /**
     * @ORM\Column(type="string", nullable=true)
     * @Assert\NotBlank()
     *
     * @var string
     */
    private $nameCarrier;

    /**
     * @ORM\ManyToOne(targetEntity="DeliveryMethod", inversedBy="shippingPrices")
     * @ORM\JoinColumn(name="delivery_method_id", referencedColumnName="id", nullable=true)
     * @var DeliveryMethod|null
     */
    private $deliveryMethod;

    /**
     * @ORM\ManyToMany(targetEntity="ShippingZone")
     * @Assert\Count( min = 1 , minMessage="Vous devez ajouter au moins une zone")
     *
     * @var Collection<ShippingZone>
     */
    private $zones;

    /**
     * @ORM\ManyToMany(targetEntity="CpCreation\VitiCore\Product\Model\ProductType")
     * @Assert\Count( min = 1 , minMessage="Vous devez ajouter au moins un type")
     * @var Collection<ProductType>
     */
    private $types;

    /**
     * @ORM\ManyToMany(targetEntity="CpCreation\VitiCore\Product\Model\ProductPriceGroup")
     * @Assert\Count( min = 1 , minMessage="Vous devez ajouter au moins un groupe")
     * @var Collection<ProductPriceGroup>
     */
    private $groups;

    public function __construct()
    {
        $this->id    = Uuid::uuid4();
        $this->types = [];
        $this->zones = new ArrayCollection();
        $this->types = new ArrayCollection();
        $this->groups = new ArrayCollection();
        $this->configGlobal = false;
    }

    /**
     * @return UuidInterface
     */
    public function getId(): UuidInterface
    {
        return $this->id;
    }

    /**
     * @return int
     */
    public function getQtyOf(): ?int
    {
        return $this->qtyOf;
    }

    /**
     * @param int $qtyOf
     */
    public function setQtyOf(int $qtyOf): void
    {
        $this->qtyOf = $qtyOf;
    }

    /**
     * @return int
     */
    public function getQtyAt(): ?int
    {
        return $this->qtyAt;
    }

    /**
     * @param int $qtyAt
     */
    public function setQtyAt(int $qtyAt): void
    {
        $this->qtyAt = $qtyAt;
    }

    /**
     * @return int
     */
    public function getPackagePrice(): ?int
    {
        return $this->packagePrice;
    }

    /**
     * @param int $packagePrice
     */
    public function setPackagePrice(int $packagePrice = null): void
    {
        $this->packagePrice = $packagePrice;
    }

    /**
     * @return int
     */
    public function getBottlePrice(): ?int
    {
        return $this->bottlePrice;
    }

    /**
     * @param int $bottlePrice
     */
    public function setBottlePrice(int $bottlePrice = null): void
    {
        $this->bottlePrice = $bottlePrice;
    }

    /**
     * @return bool
     */
    public function isConfigGlobal(): bool
    {
        return $this->configGlobal;
    }

    /**
     * @param bool $configGlobal
     */
    public function setConfigGlobal(bool $configGlobal): void
    {
        $this->configGlobal = $configGlobal;
    }

    /**
     * @return float
     */
    public function getPrivateMargePercent(): ?float
    {
        return $this->privateMargePercent;
    }

    /**
     * @param float $privateMargePercent
     */
    public function setPrivateMargePercent(float $privateMargePercent = null)
    {
        $this->privateMargePercent = $privateMargePercent;
    }

    /**
     * @return int
     */
    public function getPrivateMarge(): ?int
    {
        return $this->privateMarge;
    }

    /**
     * @param int $privateMarge
     */
    public function setPrivateMarge(int $privateMarge = null)
    {
        $this->privateMarge = $privateMarge;
    }

    /**
     * @return string
     */
    public function getNameCarrier(): ?string
    {
        return $this->nameCarrier;
    }

    /**
     * @param string $nameCarrier
     */
    public function setNameCarrier(?string $nameCarrier): void
    {
        $this->nameCarrier = $nameCarrier;
    }

    /**
     * @return DeliveryMethod|null
     */
    public function getDeliveryMethod(): ?DeliveryMethod
    {
        return $this->deliveryMethod;
    }

    /**
     * @param DeliveryMethod|null $deliveryMethod
     */
    public function setDeliveryMethod(?DeliveryMethod $deliveryMethod): void
    {
        $this->deliveryMethod = $deliveryMethod;
    }

    /**
     * @param ProductType $productType
     * @return $this
     */
    public function addType(ProductType $productType)
    {
        $this->types[] = $productType;

        return $this;
    }

    /**
     * @param ProductType $productType
     */
    public function removeType(ProductType $productType)
    {
        $this->types->removeElement($productType);
    }

    /**
     * @return Collection<ProductType>
     */
    public function getTypes(): Collection
    {
        return $this->types;
    }

    /**
     * @param ShippingZone $zone
     * @return $this
     */
    public function addZone(ShippingZone $zone)
    {
        $this->zones[] = $zone;

        return $this;
    }

    /**
     * @param ShippingZone $zone
     */
    public function removeZone(ShippingZone $zone)
    {
        $this->zones->removeElement($zone);
    }

    /**
     * @return Collection<ShippingZone>
     */
    public function getZones(): Collection
    {
        return $this->zones;
    }

    /**
     * @param ProductPriceGroup $group
     * @return $this
     */
    public function addGroup(ProductPriceGroup $group)
    {
        $this->groups[] = $group;

        return $this;
    }

    /**
     * @param ProductPriceGroup $group
     */
    public function removeGroup(ProductPriceGroup $group)
    {
        $this->groups->removeElement($group);
    }

    /**
     * @return Collection<ProductPriceGroup>
     */
    public function getGroups(): Collection
    {
        return $this->groups;
    }

    /**
     * @param \Symfony\Component\Validator\Context\ExecutionContextInterface $context
     * @Assert\Callback()
     */
    public function validate(ExecutionContextInterface $context)
    {
        if (is_null($this->getPackagePrice()) && is_null($this->getBottlePrice())) {
            $message = 'Veuillez ajouter un prix';
            $context->buildViolation($message)
                    ->atPath('bottlePrice')
                    ->addViolation();
        }
    }
}
