<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Cart\Model;

use CpCreation\VitiCore\Behavior\HasPayment;
use CpCreation\VitiCore\Behavior\HasTimestamp;
use CpCreation\VitiCore\Behavior\Impl\Timestamp;
use CpCreation\VitiCore\Product\Model\Product;
use CpCreation\VitiCore\Shop\Model\DeliveryMethod;
use CpCreation\VitiCore\User\Model\User;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\Common\Collections\Criteria;
use Doctrine\ORM\Mapping as ORM;
use Ramsey\Uuid\Uuid;
use Ramsey\Uuid\UuidInterface;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * Class Cart
 *
 * @ORM\Entity(repositoryClass="CpCreation\VitiCore\Cart\Repository\CartRepository")
 * @ORM\Table(name="cart_cart")
 * @ORM\HasLifecycleCallbacks()
 */
class Cart implements HasTimestamp, HasPayment
{
    use Timestamp;

    /**
     * @ORM\Id()
     * @ORM\Column(type="uuid")
     * @ORM\GeneratedValue(strategy="NONE")
     *
     * @var UuidInterface
     */
    private UuidInterface $id;

    /**
     * @ORM\Column(type="text")
     *
     * @var string
     */
    private string $token;

    /**
     * @ORM\Column(type="integer", nullable=true)
     *
     * @var integer|null
     */
    private ?int $shipping = null;

    /**
     * @ORM\Column(type="string", nullable=true)
     *
     * @var string | null
     */
    private ?string $shippingName = null;

    /**
     * @ORM\Column(type="string", nullable=true)
     *
     * @var string|null
     */
    private ?string $groupCode = null;

    /**
     * @ORM\Column(type="json", nullable=true)
     *
     * @var string|null
     */
    private ?string $deliveryMethodJson = null;

    /**
     * @ORM\Column(type="string", nullable=true)
     *
     * @var string|null
     */
    private ?string $promo = null;

    /**
     * @ORM\Column(type="uuid", nullable=true)
     *
     * @var UuidInterface|null
     */
    private ?UuidInterface $promoId = null;

    /**
     * @ORM\Column(type="integer", nullable=true)
     *
     * @var float | null
     */
    private ?float $promoValueAmount = null;

    /**
     * @ORM\Column(type="decimal", precision=3, scale=2, nullable=true)
     *
     * @var float | null
     */
    private ?float $promoValuePercent = null;

    /**
     * @ORM\Column(type="boolean", options={"default" : 0})
     *
     * @var boolean
     */
    private bool $promoWithShippingFees = false;
    /**
     * @ORM\Column(type="integer", nullable=true)
     *
     * @var integer|null
     */
    private ?int $degressiveValue = null;

    /**
     * @ORM\Column(type="boolean", nullable=true)
     *
     * @var boolean
     */
    private bool $viewHT = false;

    /**
     * @ORM\Column(type="decimal", precision=3, scale=2, nullable=true)
     *
     * @var float|null
     */
    private ?float $tax = null;

    /**
     * @ORM\Column(type="text", nullable=true)
     *
     * @var string| null
     */
    private ?string $comment = null;

    /**
     * @ORM\ManyToOne(targetEntity="CpCreation\VitiCore\Shop\Model\DeliveryMethod")
     * @ORM\JoinColumn(referencedColumnName="id", nullable=true)
     *
     * @var DeliveryMethod|null
     *
     */
    private ?DeliveryMethod $deliveryMethod = null;

    /**
     * @ORM\OneToOne(targetEntity="CpCreation\VitiCore\Cart\Model\CartCheckout", mappedBy="cart", cascade={"persist"})
     * @Assert\Valid()
     *
     * @var CartCheckout
     */
    private CartCheckout $cartCheckout;

    /**
     * @var CartItem[]
     *
     * @ORM\OneToMany(targetEntity="CartItem", mappedBy="cart", cascade={"persist","remove"}, orphanRemoval=true)
     * @ORM\OrderBy({"createdAt" = "ASC"})
     * @Assert\Valid()
     */
    private Collection $items;

    /**
     * @var InvoiceAddress | null
     *
     * @ORM\OneToOne(targetEntity="InvoiceAddress", cascade={"persist"})
     * @Assert\Valid()
     */
    private ?InvoiceAddress $invoiceAddress = null;

    /**
     * @var DeliveryAddress | null
     *
     * @ORM\OneToOne(targetEntity="DeliveryAddress", cascade={"persist"})
     * @Assert\Valid()
     */
    private ?DeliveryAddress $deliveryAddress = null;

    /**
     * @var User
     *
     * @ORM\ManyToOne(targetEntity="CpCreation\VitiCore\User\Model\User", inversedBy="carts")
     */
    private ?User $user = null;

    /**
     * @var Invoice
     *
     * @ORM\OneToOne(targetEntity="Invoice", inversedBy="cart")
     */
    private ?Invoice $invoice = null;

    public function __construct()
    {
        $this->id = Uuid::uuid4();
        $this->items = new ArrayCollection();
        $this->token = strtoupper(bin2hex(random_bytes(3)));
        $this->promoWithShippingFees = false;
    }

    /**
     * @return UuidInterface
     */
    public function getId(): UuidInterface
    {
        return $this->id;
    }

    /**
     * @return string
     */
    public function getToken(): ?string
    {
        return $this->token;
    }

    /**
     * @throws \Exception
     */
    public function setToken(): void
    {
        $this->token = strtoupper(bin2hex(random_bytes(3)));
    }

    /**
     * @return int
     */
    public function getShipping(): ?int
    {
        return $this->shipping;
    }

    /**
     * @param int $shipping
     */
    public function setShipping(int $shipping = null): void
    {
        $this->shipping = $shipping;
    }

    /**
     * @return string
     */
    public function getShippingName(): ?string
    {
        return $this->shippingName;
    }

    /**
     * @param string $shippingName
     */
    public function setShippingName(?string $shippingName): void
    {
        $this->shippingName = $shippingName;
    }

    /**
     * @return string
     */
    public function getGroupCode(): ?string
    {
        return $this->groupCode;
    }

    /**
     * @param string $groupCode
     */
    public function setGroupCode(?string $groupCode): void
    {
        $this->groupCode = $groupCode;
    }

    /**
     * @return string
     */
    public function getDeliveryMethodJson(): ?string
    {
        return $this->deliveryMethodJson;
    }

    /**
     * @param string $deliveryMethodJson
     */
    public function setDeliveryMethodJson(string $deliveryMethodJson = null)
    {
        $this->deliveryMethodJson = $deliveryMethodJson;
    }


    /**
     * @return string
     */
    public function getPromo(): ?string
    {
        return $this->promo;
    }

    /**
     * @param string $promo
     */
    public function setPromo(?string $promo = null): void
    {
        $this->promo = strtoupper($promo);
    }

    /**
     * @return UuidInterface
     */
    public function getPromoId(): ?UuidInterface
    {
        return $this->promoId;
    }

    /**
     * @param UuidInterface $promoId
     */
    public function setPromoId(?UuidInterface $promoId): void
    {
        $this->promoId = $promoId;
    }

    /**
     * @return float|null
     */
    public function getPromoValueAmount(): ?float
    {
        return $this->promoValueAmount;
    }

    /**
     * @param float|null $promoValueAmount
     */
    public function setPromoValueAmount(?float $promoValueAmount): void
    {
        $this->promoValueAmount = $promoValueAmount;
    }

    /**
     * @return float|null
     */
    public function getPromoValuePercent(): ?float
    {
        return $this->promoValuePercent;
    }

    /**
     * @param float|null $promoValuePercent
     */
    public function setPromoValuePercent(?float $promoValuePercent): void
    {
        $this->promoValuePercent = $promoValuePercent;
    }

    /**
     * @return bool
     */
    public function isPromoWithShippingFees(): bool
    {
        return $this->promoWithShippingFees;
    }

    /**
     * @param bool $promoWithShippingFees
     */
    public function setPromoWithShippingFees(bool $promoWithShippingFees): void
    {
        $this->promoWithShippingFees = $promoWithShippingFees;
    }

    /**
     * @return int
     */
    public function getDegressiveValue(): ?int
    {
        return $this->degressiveValue;
    }

    /**
     * @param int $degressiveValue
     */
    public function setDegressiveValue(?int $degressiveValue = null): void
    {
        $this->degressiveValue = $degressiveValue;
    }

    /**
     * @return bool
     */
    public function isViewHT(): ?bool
    {
        return $this->viewHT;
    }

    /**
     * @param bool $viewHT
     */
    public function setViewHT(bool $viewHT = null): void
    {
        $this->viewHT = $viewHT;
    }

    /**
     * @return float
     */
    public function getTax(): ?float
    {
        return $this->tax;
    }

    /**
     * @param float|null $tax
     */
    public function setTax(?float $tax = null): void
    {
        $this->tax = $tax;
    }

    /**
     * @return string
     */
    public function getComment(): ?string
    {
        return $this->comment;
    }


    public function setComment(?string $comment = null): void
    {
        $this->comment = $comment;
    }


    public function getDeliveryMethod(): ?DeliveryMethod
    {
        return $this->deliveryMethod;
    }


    public function setDeliveryMethod(?DeliveryMethod $deliveryMethod = null)
    {
        $this->deliveryMethod = $deliveryMethod;
    }


    public function getCartCheckout(): ?CartCheckout
    {
        return $this->cartCheckout;
    }

    public function setCartCheckout(?CartCheckout $cartCheckout = null)
    {
        $cartCheckout->setCart($this);
        $this->cartCheckout = $cartCheckout;
    }

    public function getUser(): ?User
    {
        return $this->user;
    }

    public function setUser(?User $user): void
    {
        $this->user = $user;
    }

    /**
     * @return int
     */
    public function getItemsTotal(): int
    {
        $total = 0;
        /** @var CartItem $item */
        foreach ($this->getItems() as $item) {
            $total = $total + $item->getTotal();
        }

        return $total;
    }

    public function getTotal(): ?int
    {

        $total = $this->getItemsTotal() + $this->getShipping();
        if ($this->getPromoValuePercent()) {
            $total = $total - ceil($this->getItemsTotal() * $this->getPromoValuePercent());
        }
        if ($this->getPromoValueAmount()) {
            $total = $total - $this->getPromoValueAmount();
        }
        if ($this->getDegressiveValue()) {
            $total = $total - $this->getDegressiveValue();
        }
        if ($this->isViewHT()) {
            return round($total * (1 + $this->getTax()), 2);
        }

        return $total;
    }

    public function getTotalWithoutTax(): float
    {
        $total = $this->getItemsTotal() + $this->getShipping();
        if ($this->getPromoValuePercent()) {
            if ($this->isPromoWithShippingFees()) {
                return $total - ceil($total * $this->getPromoValuePercent());
            } else {
                return $total - ceil($this->getItemsTotal() * $this->getPromoValuePercent());
            }
        }
        if ($this->getPromoValueAmount()) {
            $total = $total - $this->getPromoValueAmount();
        }
        if ($this->getDegressiveValue()) {
            $total = $total - $this->getDegressiveValue();
        }

        return $total;
    }

    public function getTotalTax(): float
    {
        $total = $this->getItemsTotal() + $this->getShipping();
        if ($this->getPromoValuePercent()) {
            if ($this->isPromoWithShippingFees()) {
                $total = $total - ceil($total * $this->getPromoValuePercent());
            } else {
                $total = $total - ceil($this->getItemsTotal() * $this->getPromoValuePercent());
            }

        }
        if ($this->getPromoValueAmount()) {
            $total = $total - $this->getPromoValueAmount();
        }
        if ($this->getDegressiveValue()) {
            $total = $total - $this->getDegressiveValue();
        }

        return round($total * $this->getTax(), 2);
    }

    public function getQuantityAllAbsolute(): int
    {
        $quantityAllAbsolu = 0;
        /** @var CartItem $item */
        foreach ($this->getItems() as $item) {
            $quantityAllAbsolu = $quantityAllAbsolu + ($item->getQuantity() * $item->getVariant()->getType()->getEquivalency());
        }

        return $quantityAllAbsolu;
    }

    /**
     * @param Product[] $products
     */
    public function getQuantityByProducts(Collection $products): int
    {
        if ($products->count() === 0) {
            return $this->getQuantityAllAbsolute();
        }

        $quantity = 0;
        /** @var CartItem $item */
        foreach ($this->getItems() as $item) {
            /** @var CartItem $selectedItem */
            foreach ($products as $selectedProduct) {
                if ($item->getVariant()->getProduct() === $selectedProduct) {
                    $quantity = $quantity + ($item->getQuantity() * $item->getVariant()->getType()->getEquivalency());
                }
            }

        }

        return $quantity;
    }

    /**
     * @return Collection<CartItem>
     */
    public function getItems(): Collection
    {
        return $this->items;
    }

    public function addItem(CartItem $item): void
    {
        $criteria = Criteria::create()
            ->setMaxResults(1)
            ->andWhere(Criteria::expr()->eq('variant', $item->getVariant()))
            ;
        $alreadyExist = $this->items->matching($criteria)->first();

        if ($alreadyExist) {
            $alreadyExist->setQuantity($alreadyExist->getQuantity() + $item->getQuantity());

            return;
        }
        $item->setCart($this);
        $this->items->add($item);
    }

    /**
     * @param CartItem $item
     */
    public function removeItem(CartItem $item): void
    {
        $this->items->removeElement($item);
    }

    public function getDeliveryAddress(): ?DeliveryAddress
    {
        return $this->deliveryAddress;
    }

    /**
     * @param DeliveryAddress $deliveryAddress
     */
    public function setDeliveryAddress(DeliveryAddress $deliveryAddress): void
    {
        $this->deliveryAddress = $deliveryAddress;
    }

    /**
     * @return InvoiceAddress
     */
    public function getInvoiceAddress(): ?InvoiceAddress
    {
        return $this->invoiceAddress;
    }

    /**
     * @param InvoiceAddress $invoiceAddress
     */
    public function setInvoiceAddress(InvoiceAddress $invoiceAddress): void
    {
        $this->invoiceAddress = $invoiceAddress;
    }

    public function getEmail(): ?string
    {
        if ($this->getInvoiceAddress()) {
            return $this->getInvoiceAddress()->getEmail();
        }

        return null;
    }

    public function getPaymentMethod(): ?PaymentMethod
    {
        if ($this->getCartCheckout()) {
            return $this->getCartCheckout()->getPaymentMethod();
        }

        return null;
    }

    public function getFirstname(): ?string
    {
        if ($this->getInvoiceAddress()) {
            return $this->getInvoiceAddress()->getFirstname();
        }

        return null;
    }

    public function getLastname(): ?string
    {
        if ($this->getInvoiceAddress()) {
            return $this->getInvoiceAddress()->getLastname();
        }

        return null;
    }

    public function getPhone(): ?string
    {
        if ($this->getInvoiceAddress()) {
            return $this->getInvoiceAddress()->getPhone();
        }

        return null;
    }

    public function getCompany(): ?string
    {
        if ($this->getInvoiceAddress()) {
            return $this->getInvoiceAddress()->getCompany();
        }

        return null;
    }

    public function getStreet(): ?string
    {
        if ($this->getInvoiceAddress()) {
            return $this->getInvoiceAddress()->getStreet();
        }

        return null;
    }

    public function getPostalCode(): ?string
    {
        if ($this->getInvoiceAddress()) {
            return $this->getInvoiceAddress()->getPostalCode();
        }

        return null;
    }

    public function getCity(): ?string
    {
        if ($this->getInvoiceAddress()) {
            return $this->getInvoiceAddress()->getCity();
        }

        return null;
    }

    public function getCountry(): ?string
    {
        if ($this->getInvoiceAddress()) {
            return $this->getInvoiceAddress()->getCountry();
        }

        return null;
    }

    public function getQuantity(): int
    {
        $quantity = 0;
        foreach ($this->getItems() as $item) {
            $quantity = $quantity + $item->getQuantity();
        }

        return $quantity;
    }

    public function getInvoice(): ?Invoice
    {
        return $this->invoice;
    }

    public function setInvoice(?Invoice $invoice): void
    {
        $this->invoice = $invoice;
    }

    public function containsGifts(): bool
    {
        /** @var CartItem $item */
        foreach ($this->items as $item) {
            if ($item->getVariant()->getType()->isGift()) {
                return true;
            }
        }
        return false;
    }
}
