<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Bridge\Symfony\Bundle\Utils;

use CpCreation\VitiCore\Bridge\Symfony\Bundle\Filesystem\OrderFilesystem;
use CpCreation\VitiCore\Cart\Model\Cart;
use CpCreation\VitiCore\Cart\Model\CartItem;
use CpCreation\VitiCore\Contact\Model\Contact;
use CpCreation\VitiCore\Contact\Repository\ContactRepository;
use CpCreation\VitiCore\Payment\Model\Payment;
use CpCreation\VitiCore\Product\Model\ProductGift;
use Symfony\Bridge\Twig\Mime\TemplatedEmail;
use Symfony\Component\Mailer\MailerInterface;
use Symfony\Component\Mime\Address;
use Symfony\Component\Mime\Email;
use Symfony\Component\Mime\Part\DataPart;
use Symfony\Component\Mime\Part\File;
use Symfony\Contracts\Translation\TranslatorInterface;
use Twig\Environment;

class Mailer
{
    private $pdf;
    private $mailer;
    private $filesystem;
    private $translator;
    private $emailToMaintenance;
    private $contactRepository;
    private $twig;

    public function __construct(Pdf                 $pdf,
                                PdfPuppeteer        $pdfPuppeteer,
                                MailerInterface     $mailer,
                                OrderFilesystem     $filesystem,
                                TranslatorInterface $translator,
                                ContactRepository   $contactRepository,
                                Environment         $twig,
                                                    $emailToMaintenance
    )
    {
        $this->pdf = $pdf;
        $this->pdfPuppeteer = $pdfPuppeteer;
        $this->mailer = $mailer;
        $this->filesystem = $filesystem;
        $this->translator = $translator;
        $this->emailToMaintenance = $emailToMaintenance;
        $this->twig = $twig;
        $this->contactRepository = $contactRepository;
    }

    public function send(Email $message)
    {
        $this->mailer->send($message);
    }

    /**
     * @param Contact $contact
     * @param Cart $cart
     * @throws \Twig\Error\LoaderError
     * @throws \Twig\Error\RuntimeError
     * @throws \Twig\Error\SyntaxError
     */
    public function sendOrder(Contact $contact, Cart $cart)
    {
        $path = $this->pdf->createPdfOrder($contact, $cart);
        $messageCustomer = (new TemplatedEmail())
            ->subject($this->translator->trans('order.mail.subject_mail', ['%company%' => $contact->getCompany()]))
            ->from(new Address($contact->getEmailFrom(), $contact->getCompany()))
            ->sender(new Address($contact->getEmailFrom(), $contact->getCompany()))
            ->returnPath(new Address($contact->getEmailFrom()))
            ->to(new Address($cart->getInvoiceAddress()->getEmail()))
            ->addPart(
                new DataPart(new File('order/' . $path), '', 'application/pdf')
            )
            ->htmlTemplate('mail/order_customer.html.twig')
            ->context([
                'contact' => $contact,
                'cart' => $cart,
                'company' => $contact->getCompany()
            ]);

        $this->mailer->send($messageCustomer);

        $path = $this->pdf->createPdfOrderAdmin($contact, $cart, 'fr');

        $message = (new TemplatedEmail())
            ->subject('Un nouveau bon de commande est arrivé ! de ' . $cart->getInvoiceAddress()->getLastname() . ' ' . $cart->getInvoiceAddress()->getFirstname())
            ->from(new Address($contact->getEmailFrom(), $contact->getCompany()))
            ->returnPath(new Address($contact->getEmailFrom()))
            ->to(new Address($contact->getEmailToShop(), $contact->getCompany()))
            ->addPart(
                new DataPart(new File('order/' . $path), '', 'application/pdf')
            )
            ->htmlTemplate('mail/order_admin.html.twig')
            ->context([
                'contact' => $contact,
                'cart' => $cart,
                'company' => $contact->getCompany()
            ]);

        $this->mailer->send($message);
    }

    public function sendPayment(Payment $payment)
    {
        /** @var Contact $contact */
        $contact = $this->contactRepository->findOne();

        $message = (new Email())
            ->subject('Paiement en ligne sur le site ' . $contact->getCompany() . ' - Référence ' . $payment->getReference())
            ->from(new Address($contact->getEmailFrom(), $contact->getCompany()))
            ->returnPath(new Address($contact->getEmailFrom()))
            ->to(new Address($contact->getEmailToShop(), $contact->getCompany()))
            ->cc(new Address($payment->getEmail(), $payment->getLastname() . ' ' . $payment->getFirstname()))
            ->html(
                'Nom : ' . $payment->getLastname() . '<br />' .
                'Prénom : ' . $payment->getFirstname() . '<br />' .
                'Adresse : ' . $payment->getStreet() . '<br />' .
                'Code postal : ' . $payment->getPostalCode() . '<br />' .
                'Ville : ' . $payment->getCity() . '<br />' .
                'Email : <a href="' . $payment->getEmail() . '">' . $payment->getEmail() . '</a><br />' .
                'Téléphone : ' . $payment->getPhone() . '<br />' .
                'Référence Commande : ' . $payment->getSlug() . '<br />' .
                'Montant : ' . number_format($payment->getAmount() / 100, 2) . ' € TTC <br />'
            )
            ->text(
                'Nom : ' . $payment->getLastname() .
                ' Prénom : ' . $payment->getFirstname() .
                ' Adresse : ' . $payment->getStreet() .
                ' Code postal : ' . $payment->getPostalCode() .
                ' Ville : ' . $payment->getCity() .
                ' Email : ' . $payment->getEmail() .
                ' Téléphone : ' . $payment->getPhone() .
                ' Référence Commande : ' . $payment->getSlug() .
                ' Montant : ' . number_format($payment->getAmount() / 100, 2) . ' € TTC'
            );

        $this->mailer->send($message);
    }

    /**
     * @param Contact $contact
     * @param Cart $cart
     * @throws \Twig\Error\LoaderError
     * @throws \Twig\Error\RuntimeError
     * @throws \Twig\Error\SyntaxError
     */
    public function senDelivery(Contact $contact, Cart $cart)
    {
        $path = $this->pdf->createPdfOrder($contact, $cart);

        $message = (new TemplatedEmail())
            ->subject($this->translator->trans(
                'delivery.mail.subject_mail', [
                '%company%' => $contact->getCompany(),
                '%ref%' => $cart->getToken(),
            ], null, $cart->getCartCheckout()->getLocale()
            ))
            ->from(new Address($contact->getEmailFrom(), $contact->getCompany()))
            ->returnPath(new Address($contact->getEmailFrom()))
            ->to(new Address($cart->getInvoiceAddress()->getEmail()))
            ->htmlTemplate('mail/delivery.html.twig')
            ->textTemplate('mail/delivery.txt.twig')
            ->addPart(
                new DataPart(new File('order/' . $path), '', 'application/pdf')
            )
            ->context([
                'contact' => $contact,
                'cart' => $cart,
                'company' => $contact->getCompany(),
            ]);

        $this->mailer->send($message);
    }

    /**
     * @param Contact $contact
     * @param Cart $cart
     * @param CartItem $item
     * @return void
     * @throws \Twig\Error\LoaderError
     * @throws \Twig\Error\RuntimeError
     * @throws \Twig\Error\SyntaxError
     */
    public function sendGiftCard(Contact $contact, Cart $cart)
    {

        $message = (new TemplatedEmail())
            ->subject($this->translator->trans(
                'gift.mail.subject_mail', [], null, $cart->getCartCheckout()->getLocale()
            ))
            ->from(new Address($contact->getEmailFrom(), $contact->getCompany()))
            ->returnPath(new Address($contact->getEmailFrom()))
            ->to(new Address($cart->getInvoiceAddress()->getEmail()))
            ->htmlTemplate('mail/gift.html.twig')
            ->textTemplate('mail/gift.txt.twig')
            ->context([
                'cart' => $cart
            ]);
        foreach ($cart->getItems() as $cartItem) {
            /** @var ProductGift $gift */
            foreach ($cartItem->getGifts() as $gift) {
                $path = $this->pdfPuppeteer->createGiftCard($gift, $contact);
                $message->addPart(
                    new DataPart(new File('upload/gift/' . $path), '', 'application/pdf')
                );
            }
        }

        $this->mailer->send($message);
    }

    /**
     * @param $exception
     * @throws \Doctrine\ORM\NonUniqueResultException
     */
    public function sendErrorConnexionApiConnectSocialInstagram($exception)
    {

        /** @var Contact $contact */
        $contact = $this->contactRepository->findOne();

        $message = (new Email())
            ->subject('Erreur Connexion à l\'api Connect Social CPCREATION ' . $contact->getCompany())
            ->from(new Address($contact->getEmailFrom(), $contact->getCompany()))
            ->to(new Address($this->emailToMaintenance))
            ->html($exception . ' ' . $contact->getCompany())
            ->text($exception . ' ' . $contact->getCompany());

        $this->mailer->send($message);
    }

    /**
     * @param $exception
     * @throws \Doctrine\ORM\NonUniqueResultException
     */
    public function sendErrorInstagram($exception)
    {

        /** @var Contact $contact */
        $contact = $this->contactRepository->findOne();

        $message = (new Email())
            ->subject('Erreur connexion à l\'api display Instagram ' . $contact->getCompany())
            ->from(new Address($contact->getEmailFrom(), $contact->getCompany()))
            ->to(new Address($this->emailToMaintenance))
            ->html($exception . ' ' . $contact->getCompany())
            ->text($exception . ' ' . $contact->getCompany());

        $this->mailer->send($message);
    }

    /**
     * @param $exception
     * @throws \Doctrine\ORM\NonUniqueResultException
     */
    public function sendErrorFacebookInstagram($exception)
    {

        /** @var Contact $contact */
        $contact = $this->contactRepository->findOne();

        $message = (new Email())
            ->subject('Erreur connexion à l\'api Facebook pour Instagram ' . $contact->getCompany())
            ->from(new Address($contact->getEmailFrom(), $contact->getCompany()))
            ->to(new Address($this->emailToMaintenance))
            ->html($exception . ' ' . $contact->getCompany())
            ->text($exception . ' ' . $contact->getCompany());

        $this->mailer->send($message);
    }

    /**
     * @param $exception
     * @throws \Doctrine\ORM\NonUniqueResultException
     */
    public function sendErrorFacebookFacebook($exception)
    {

        /** @var Contact $contact */
        $contact = $this->contactRepository->findOne();

        $message = (new Email())
            ->subject('Erreur connexion à l\'api Facebook pour Facebook ' . $contact->getCompany())
            ->from(new Address($contact->getEmailFrom(), $contact->getCompany()))
            ->to(new Address($this->emailToMaintenance))
            ->html($exception . ' ' . $contact->getCompany())
            ->text($exception . ' ' . $contact->getCompany());

        $this->mailer->send($message);
    }

    /**
     * @param $exception
     * @throws \Doctrine\ORM\NonUniqueResultException
     */
    public function sendErrorException($exception)
    {

        /** @var Contact $contact */
        $contact = $this->contactRepository->findOne();
        $message = (new Email())
            ->subject('Erreur ' . $contact->getCompany())
            ->from(new Address($contact->getEmailFrom(), $contact->getCompany()))
            ->to(new Address($this->emailToMaintenance))
            ->html($exception . ' ' . $contact->getCompany())
            ->text($exception . ' ' . $contact->getCompany());

        $this->mailer->send($message);
    }

    /**
     * @param $response
     * @throws \Doctrine\ORM\NonUniqueResultException
     */
    public function sendErrorValidReturnPayment($response)
    {

        /** @var Contact $contact */
        $contact = $this->contactRepository->findOne();

        /** @var Contact $contact */
        $contact = $this->contactRepository->findOne();
        $message = (new Email())
            ->subject('Erreur Validation Paiement - retour de banque ' . $contact->getCompany())
            ->from(new Address($contact->getEmailFrom(), $contact->getCompany()))
            ->to(new Address($this->emailToMaintenance))
            ->html($response)
            ->text($response);

        $this->mailer->send($message);
    }

    /**
     * @param $exception
     * @throws \Doctrine\ORM\NonUniqueResultException
     */
    public function sendErrorIpLocate($exception, $ip)
    {

        /** @var Contact $contact */
        $contact = $this->contactRepository->findOne();

        $message = (new Email())
            ->subject('Erreur location de l\'adresse IP ' . $ip . ' ' . $contact->getCompany())
            ->from(new Address($contact->getEmailFrom(), $contact->getCompany()))
            ->to(new Address($this->emailToMaintenance))
            ->html(  $exception . ' ' . $contact->getCompany())
            ->text(  $exception . ' ' . $contact->getCompany());

        $this->mailer->send($message);
    }
}
