<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Admin;

use CpCreation\VitiCore\Award\Model\Award;
use CpCreation\VitiCore\Award\Repository\AwardRepository;
use CpCreation\VitiCore\Bridge\Symfony\Bundle\Form\Admin\AwardType;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Security;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;

/**
 * Class AwardController
 *
 * @package CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Admin
 * @Security("is_granted('ROLE_ADMIN') or is_granted('ROLE_AWARD')")
 * @Route("/recompenses")
 */
class AwardController extends AbstractController
{
    private $awardRepository;

    public function __construct(AwardRepository $awardRepository)
    {
        $this->awardRepository = $awardRepository;
    }

    /**
     * @route("", name="admin_award")
     */
    public function indexAction(): Response
    {
        $awards = $this->awardRepository->findByCreatedAdminAt('DESC');

        return $this->render('admin/award/index.html.twig', [
            "items" => $awards
        ]);
    }

    /**
     * @param Request $request
     * @Route("/new", name="admin_award_new")
     *
     * 
     */
    public function newAction(Request $request): Response|RedirectResponse
    {

        $form = $this->createForm(AwardType::class);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            /** @var Award $data */
            $data = $form->getData();
            $this->awardRepository->save($data);

            $this->addFlash('alert-success', 'La récompense ' . $data->getTitle() . ' a été ajouté avec succès');

            return $this->redirectToRoute('admin_award');
        }

        return $this->render('admin/award/new.html.twig', [
            "form" => $form->createView(),
        ]);
    }

    /**
     * @param Award $award
     * @param Request $request
     * @Route("/{id}/edit", name="admin_award_edit")
     *
     * 
     */
    public function editAction(Award $award, Request $request): Response|RedirectResponse
    {
        $form = $this->createForm(AwardType::class, $award);
        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            /** @var Award $data */
            $data = $form->getData();
            $data->setUpdatedAt(new \DateTime()); // Forcer Update de doctrine
            $this->awardRepository->save($data);
            $this->addFlash('alert-success', 'La récompense ' . $award->getTitle() . ' a été modifié avec succès');

            return $this->redirectToRoute('admin_award');
        }

        return $this->render('admin/award/new.html.twig', [
            "item" => $award,
            "form" => $form->createView(),
        ]);
    }

    /**
     * @param Award $award
     * @Route("/{id}/delete", name="admin_award_delete")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function deleteAction(Award $award): RedirectResponse
    {
        $this->awardRepository->delete($award);
        $this->addFlash('alert-success', 'La récompense ' . $award->getTitle() . ' a été supprimé avec succès.');

        return $this->redirectToRoute('admin_award');
    }
}
