<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Bridge\Symfony\Bundle\Form\Site;

use CpCreation\VitiCore\Cart\Model\DeliveryAddress;
use CpCreation\VitiCore\Product\Model\ProductPriceGroup;
use CpCreation\VitiCore\Product\Repository\ProductPriceCodeRepository;
use CpCreation\VitiCore\Product\Repository\ProductPriceGroupRepository;
use CpCreation\VitiCore\Shop\Model\ShippingCountry;
use CpCreation\VitiCore\Shop\Repository\ShippingCountryRepository;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\Extension\Core\Type\CheckboxType;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\Extension\Core\Type\EmailType;
use Symfony\Component\Form\Extension\Core\Type\TelType;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\Form\FormEvent;
use Symfony\Component\Form\FormEvents;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Session\SessionInterface;
use Symfony\Component\Intl\Countries;
use Symfony\Component\OptionsResolver\OptionsResolver;

class DeliveryAddressType extends AbstractType
{
    private $countriesWithZone;

    public function __construct(
        ShippingCountryRepository $shippingCountryRepository,
        SessionInterface $session
    ) {
        $group     = $session->get("groupPrice");
        $countries = $shippingCountryRepository->findCountriesAssignedByGroup($group);
        /** @var ShippingCountry $country */
        foreach ($countries as $country) {
            $this->countriesWithZone[$country->getCode()] = $country->getCode();
        }
    }

    public function buildForm(FormBuilderInterface $builder, array $options)
    {
        $builder
            ->add(
                'lastname', TextType::class, [
                    'label'    => 'shop.form.lastname',
                    'attr'     => [
                        'placeholder'  => 'shop.form.lastname',
                        'autocomplete' => 'family-name',
                    ],
                    'required' => true,
                ]
            )
            ->add(
                'firstname', TextType::class, [
                    'label'    => 'shop.form.firstname',
                    'attr'     => [
                        'placeholder'  => 'shop.form.firstname',
                        'autocomplete' => 'name',
                    ],
                    'required' => true,
                ]
            )
            ->add(
                'street', TextType::class, [
                    'label'    => 'shop.form.address',
                    'attr'     => [
                        'placeholder'  => 'shop.form.address',
                        'autocomplete' => 'nope',
                    ],
                    'required' => true,
                ]
            )
            ->add(
                'postalCode', TextType::class, [
                    'label'    => 'shop.form.postal_code',
                    'attr'     => [
                        'placeholder'  => 'shop.form.postal_code',
                        'autocomplete' => 'nope',
                    ],
                    'required' => true,
                ]
            )
            ->add(
                'city', TextType::class, [
                    'label'    => 'shop.form.city',
                    'attr'     => [
                        'placeholder'  => 'shop.form.city',
                        'autocomplete' => 'nope',
                    ],
                    'required' => true,
                ]
            )
            ->add(
                'country', ChoiceType::class, [
                    'label'        => 'shop.form.country',
                    'required'     => true,
                    'choices'      => $this->countriesWithZone,
                    'choice_label' => function ($choice, $key, $value) {
                        return Countries::getName($value);
                    },
                    'attr'         => [
                        'separator'    => '=====',
                        'autocomplete' => 'nope',
                    ],
                ]
            )
            ->add(
                'email', EmailType::class, [
                    'label'    => 'shop.form.email',
                    'attr'     => [
                        'placeholder'  => 'shop.form.email',
                        'autocomplete' => 'email',
                    ],
                    'required' => false,
                ]
            )
            ->add(
                'phone', TelType::class, [
                    'label'    => 'shop.form.phone',
                    'attr'     => [
                        'placeholder'  => 'shop.form.phone',
                        'autocomplete' => 'tel',
                    ],
                    'required' => false,
                ]
            )
            ->add(
                'differentAddress', CheckboxType::class, [
                    'label'    => 'shop.different_address_text',
                    'required' => false,
                ]
            );

        $builder->addEventListener(
            FormEvents::PRE_SET_DATA,
            [$this, 'onPreSetData']
        );

    }

    public function onPreSetData(FormEvent $event)
    {
        $form = $event->getForm();
        $data = $event->getData();
        if (!$data) {
            $form->remove('country');
            $form->add(
                'country', ChoiceType::class, [
                    'label'        => 'shop.form.country',
                    'required'     => true,
                    'choices'      => $this->countriesWithZone,
                    'data'         => 'FR',
                    'choice_label' => function ($choice, $key, $value) {
                        return Countries::getName($value);
                    },
                    'attr'         => [
                        'autocomplete' => 'nope',
                    ],
                ]
            );

        }
    }

    public function configureOptions(OptionsResolver $resolver)
    {
        $resolver->setDefaults(
            [
                "data_class" => DeliveryAddress::class,
            ]
        );
    }
}
