<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Admin;

use CpCreation\VitiCore\Bridge\Symfony\Bundle\Form\Admin\PaymentMethodFlow;
use CpCreation\VitiCore\Cart\Model\PaymentMethod;
use CpCreation\VitiCore\Cart\Repository\PaymentMethodRepository;
use CpCreation\VitiCore\Product\Model\Product;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\IsGranted;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Annotation\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;

/**
 * @IsGranted({"ROLE_ADMIN"})
 * Class PaymentMethodController
 *
 * @package CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Admin
 * @Route("/methode-paiement")
 */
class PaymentMethodController extends AbstractController
{
    private $paymentMethodRepository;
    private $paymentMethodFlow;

    public function __construct(PaymentMethodRepository $paymentMethodRepository, PaymentMethodFlow $paymentMethodFlow)
    {
        $this->paymentMethodRepository = $paymentMethodRepository;
        $this->paymentMethodFlow = $paymentMethodFlow;
    }
    /**
     * @route("", name="admin_payment")
     * @Template("@CpCreationVitiCore/admin/payment_method/index.html.twig")
     */
    public function indexAction()
    {
        $payments = $this->paymentMethodRepository->findByPositionOrderAdmin();

        return [
            "items" => $payments,
        ];
    }

    /**
     * @param Request $request
     * @Route("/new", name="admin_payment_new")
     * @Template("@CpCreationVitiCore/admin/payment_method/new.html.twig")
     *
     * @return array|\Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function newAction(Request $request)
    {
        $formData = new PaymentMethod();
        $flow = $this->paymentMethodFlow;
        $flow->bind($formData);

        // form of the current step
        $form = $flow->createForm();
        if ($flow->isValid($form)) {
            $flow->saveCurrentStepData($form);
            if ($flow->nextStep()) {
                $form = $flow->createForm();
            } else {
                $data = $form->getData();
                $this->paymentMethodRepository->save($data);
                $this->addFlash('alert-success', 'La méthode de paiement '.$data->getMethod().' a été ajouté avec succès');
                $flow->reset(); // remove step data from the session

                return $this->redirectToRoute('admin_payment');
            }
        }

        return [
            'form'  => $form->createView(),
            'flow'  => $flow
        ];
    }

    /**
     * @param PaymentMethod $method
     * @param Request $request
     * @Route("/{id}/edit", name="admin_payment_edit")
     * @Template("@CpCreationVitiCore/admin/payment_method/edit.html.twig")
     *
     * @return array|\Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function editAction(PaymentMethod $method, Request $request)
    {
        $flow = $this->paymentMethodFlow;
        $flow->bind($method);

        // form of the current step
        $form = $flow->createForm();
        if ($flow->isValid($form)) {
            $flow->saveCurrentStepData($form);
            if ($flow->nextStep()) {
                $form = $flow->createForm();
            } else {
                $data = $form->getData();
                $this->paymentMethodRepository->save($data);
                $this->addFlash('alert-success', 'La méthode de paiement '.$data->getMethod().' a été modifié avec succès');
                $flow->reset(); // remove step data from the session

                return $this->redirectToRoute('admin_payment');
            }
        }

        return [
            'form'  => $form->createView(),
            'item'  => $method,
            'flow'  => $flow
        ];
    }

    /**
     * @param PaymentMethod $method
     * @Route("/{id}/disable", name="admin_payment_disable")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function disableAction(PaymentMethod $method)
    {
        $method->setDisabled(1);
        $this->paymentMethodRepository->save($method);
        $this->addFlash('alert-success', 'Le mode de paiement  a été dépublié.');

        return $this->redirectToRoute('admin_payment');
    }

    /**
     * @param PaymentMethod $method
     * @Route("/{id}/enable", name="admin_payment_enable")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function enableAction(PaymentMethod $method)
    {
        $method->setDisabled(0);
        $this->paymentMethodRepository->save($method);
        $this->addFlash('alert-success', 'Le mode de paiement a été publié.');

        return $this->redirectToRoute('admin_payment');
    }

    /**
     * @param Product $product
     * @Route("/{id}/moveUp", name="admin_payment_move_up")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function moveUpAction(PaymentMethod $method)
    {
        $method->setPosition($method->getPosition() - 1);
        $this->paymentMethodRepository->save($method);
        $this->addFlash('alert-success', 'Le mode de paiement '.$method->getMethod().' a été remonté.');

        return $this->redirectToRoute('admin_payment');
    }

    /**
     * @param Product $product
     * @Route("/{id}/moveDown", name="admin_payment_move_down")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function moveDownAction(PaymentMethod $method)
    {
        $method->setPosition($method->getPosition() + 1);
        $this->paymentMethodRepository->save($method);
        $this->addFlash('alert-success', 'Le mode de paiement '.$method->getMethod().' a été descendu.');

        return $this->redirectToRoute('admin_payment');
    }
    /**
     * @param PaymentMethod $method
     * @Route("/{id}/delete", name="admin_payment_delete")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function deleteAction(PaymentMethod $method)
    {
        $this->paymentMethodRepository->softDelete($method);
        $this->addFlash('alert-success', 'Le mode de paiement '.$method->getMethod().' a été supprimé avec succès.');

        return $this->redirectToRoute('admin_payment');
    }
}
