<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Admin;

use CpCreation\VitiCore\Bridge\Symfony\Bundle\Form\Admin\NewsletterType;
use CpCreation\VitiCore\Newsletter\Model\Newsletter;
use CpCreation\VitiCore\Newsletter\Repository\NewsletterRepository;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\IsGranted;
use Symfony\Component\Routing\Annotation\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;

/**
 * @IsGranted({"ROLE_ADMIN","ROLE_NEWSLETTER"})
 * Class NewsletterController
 *
 * @package CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Admin
 * @Route("/newsletter")
 */
class NewsletterController extends AbstractController
{
    private $newsletterRepository;

    public function __construct(NewsletterRepository $newsletterRepository)
    {
        $this->newsletterRepository = $newsletterRepository;
    }

    /**
     * @route("", name="admin_newsletter")
     * @Template("@CpCreationVitiCore/admin/newsletter/index.html.twig")
     */
    public function indexAction()
    {
        $newsletters = $this->newsletterRepository->findBy([], ['createdAt' => 'DESC']);

        return [
            "items" => $newsletters,
        ];
    }

    /**
     * @param Request $request
     * @Route("/new", name="admin_newsletter_new")
     * @Template("@CpCreationVitiCore/admin/newsletter/new.html.twig")
     *
     * @return array|\Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function newAction(Request $request)
    {

        $form = $this->createForm(NewsletterType::class);
        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            /** @var Newsletter $data */
            $data = $form->getData();
            $this->newsletterRepository->save($data);
            $this->addFlash('alert-success', 'La newsletter '.$data->getTitle().' a été ajouté avec succès');

            return $this->redirectToRoute('admin_newsletter');

        }

        return [
            "form" => $form->createView(),
        ];
    }

    /**
     * @param Newsletter $newsletter
     * @param Request $request
     * @Route("/{id}/edit", name="admin_newsletter_edit")
     * @Template("@CpCreationVitiCore/admin/newsletter/edit.html.twig")
     *
     * @return array|\Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function editAction(Newsletter $newsletter, Request $request)
    {
        $form = $this->createForm(NewsletterType::class, $newsletter);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            /** @var Newsletter $data */
            $data = $form->getData();
            $data->setUpdatedAt(new \DateTime()); // Forcer Update de doctrine
            $this->newsletterRepository->save($data);
            $this->addFlash('alert-success', 'La newsletter '.$newsletter->getTitle().' a été modifié avec succès');

            return $this->redirectToRoute('admin_newsletter');

        }

        return [
            "item" => $newsletter,
            "form" => $form->createView(),
        ];
    }

    /**
     * @param Newsletter $newsletter
     * @Route("/{id}/delete", name="admin_newsletter_delete")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function deleteAction(Newsletter $newsletter)
    {
        $this->newsletterRepository->delete($newsletter);
        $this->addFlash('alert-success', 'La newsletter '.$newsletter->getTitle().' a été supprimé avec succès.');

        return $this->redirectToRoute('admin_newsletter');
    }
}
