<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Admin;

use CpCreation\VitiCore\Bridge\Symfony\Bundle\Form\Admin\LibraryType;
use CpCreation\VitiCore\Library\Model\Library;
use CpCreation\VitiCore\Library\Repository\LibraryRepository;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\IsGranted;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Annotation\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;

/**
 * @IsGranted({"ROLE_ADMIN"})
 * Class LibraryController
 * @package CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Admin
 * @Route("/library")
 */
class LibraryController extends AbstractController
{
    private $libraryRepository;

    public function __construct(LibraryRepository $libraryRepository)
    {
        $this->libraryRepository = $libraryRepository;
    }

    /**
     * @Route("", name="admin_library")
     * @Template("@CpCreationVitiCore/admin/library/index.html.twig")
     */
    public function indexAction()
    {
        $libraries = $this->libraryRepository->findBy([], ['position' => 'ASC']);

        return [
            "items" => $libraries
        ];
    }

    /**
     * @param Request $request
     * @Route("/new", name="admin_library_new")
     * @Template("@CpCreationVitiCore/admin/library/new.html.twig")
     *
     * @return array|\Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function newAction(Request $request)
    {
        $form = $this->createForm(LibraryType::class);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            /** @var Library $data */
            $data = $form->getData();
            $this->libraryRepository->save($data);
            $this->addFlash('alert-success', 'La photo '.$data->getTitle().' a été ajouté avec succès');

            return $this->redirectToRoute('admin_library');
        }

        return [
            "form" => $form->createView(),
        ];
    }

    /**
     * @param Library $library
     * @param Request $request
     * @Route("/{id}/edit", name="admin_library_edit")
     * @Template("@CpCreationVitiCore/admin/library/edit.html.twig")
     *
     * @return array|\Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function editAction(Library $library, Request $request){
        $form = $this->createForm(LibraryType::class, $library);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            /** @var Library $data */
            $data = $form->getData();
            $this->libraryRepository->save($data);
            $this->addFlash('alert-success', 'La photo '.$library->getTitle().' a été modifié avec succès');

            return $this->redirectToRoute('admin_library');
        }

        return [
            "item" => $library,
            "form" => $form->createView(),
        ];
    }

    /**
     * @param Library $library
     * @Route("/{id}/delete", name="admin_library_delete")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function deleteAction(Library $library)
    {
        $this->libraryRepository->delete($library);
        $this->addFlash('alert-success', 'La photo '.$library->getTitle().' a été supprimé avec succès.');

        return $this->redirectToRoute('admin_library');
    }

    /**
     * @param Library $library
     * @Route("/{id}/moveUp", name="admin_library_move_up")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function moveUpAction(Library $library)
    {
        $library->setPosition($library->getPosition() - 1);
        $this->libraryRepository->save($library);
        $this->addFlash('alert-success', 'La photo '.$library->getTitle().' a été remonté.');

        return $this->redirectToRoute('admin_library');
    }

    /**
     * @param Library $library
     * @Route("/{id}/moveDown", name="admin_library_move_down")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function moveDownAction(Library $library)
    {
        $library->setPosition($library->getPosition() + 1);
        $this->libraryRepository->save($library);
        $this->addFlash('alert-success', 'La photo '.$library->getTitle().' a été descendu.');

        return $this->redirectToRoute('admin_library');
    }
}