<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Admin;

use CpCreation\VitiCore\Bridge\Symfony\Bundle\Form\Admin\HistoryType;
use CpCreation\VitiCore\History\Model\History;
use CpCreation\VitiCore\History\Model\HistoryImage;
use CpCreation\VitiCore\History\Repository\HistoryRepository;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\IsGranted;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Annotation\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;

/**
 * @IsGranted({"ROLE_ADMIN"})
 * Class HistoryController
 * @package CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Admin
 * @Route("/history")
 */
class HistoryController extends AbstractController
{
    private $historyRepository;

    public function __construct(HistoryRepository $historyRepository)
    {
        $this->historyRepository = $historyRepository;
    }

    /**
     * @Route("", name="admin_history")
     * @Template("@CpCreationVitiCore/admin/history/index.html.twig")
     */
    public function indexAction()
    {
        $histories = $this->getDoctrine()->getRepository(History::class)->findBy([], ['position' => 'ASC']);

        return [
            "items" => $histories
        ];
    }

    /**
     * @param Request $request
     * @Route("/new", name="admin_history_new")
     * @Template("@CpCreationVitiCore/admin/history/new.html.twig")
     *
     * @return array|\Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function newAction(Request $request)
    {
        $form = $this->createForm(HistoryType::class);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            /** @var History $data */
            $data = $form->getData();
            $this->historyRepository->save($data);
            $this->addFlash('alert-success', 'L\'histoire '.$data->getTitle().' a été ajouté avec succès');

            return $this->redirectToRoute('admin_history');
        }

        return [
            "form" => $form->createView(),
        ];
    }

    /**
     * @param History $history
     * @param Request $request
     * @Route("/{id}/edit", name="admin_history_edit")
     * @Template("@CpCreationVitiCore/admin/history/edit.html.twig")
     *
     * @return array|\Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function editAction(History $history, Request $request){
        $form = $this->createForm(HistoryType::class, $history);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            /** @var History $data */
            $data = $form->getData();
            $data->setUpdatedAt(new \DateTime()); // Forcer Update de doctrine
            /** @var HistoryImage $image */
            foreach ($data->getImages() as $key => $image)
            {
                $image->setHistory($history);
            }
            $this->historyRepository->save($data);
            $this->addFlash('alert-success', 'L\'histoire '.$history->getTitle().' a été modifié avec succès');

            return $this->redirectToRoute('admin_history');
        }

        return [
            "item" => $history,
            "form" => $form->createView(),
        ];
    }

    /**
     * @param History $history
     * @Route("/{id}/delete", name="admin_history_delete")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function deleteAction(History $history)
    {
        $this->historyRepository->delete($history);
        $this->addFlash('alert-success', 'L\'histoire '.$history->getTitle().' a été supprimé avec succès.');

        return $this->redirectToRoute('admin_history');
    }

    /**
     * @param History $history
     * @Route("/{id}/moveUp", name="admin_history_move_up")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function moveUpAction(History $history)
    {
        $history->setPosition($history->getPosition() - 1);
        $this->historyRepository->save($history);
        $this->addFlash('alert-success', 'L\'histoire '.$history->getTitle().' a été remonté.');

        return $this->redirectToRoute('admin_history');
    }

    /**
     * @param History $history
     * @Route("/{id}/moveDown", name="admin_history_move_down")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function moveDownAction(History $history)
    {
        $history->setPosition($history->getPosition() + 1);
        $this->historyRepository->save($history);
        $this->addFlash('alert-success', 'L\'histoire '.$history->getTitle().' a été descendu.');

        return $this->redirectToRoute('admin_history');
    }
}