<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Admin;

use CpCreation\VitiCore\Bridge\Symfony\Bundle\Form\Admin\DeliveryMethodType;
use CpCreation\VitiCore\Shop\Model\DeliveryMethod;
use CpCreation\VitiCore\Shop\Model\DeliveryMethodCategory;
use CpCreation\VitiCore\Shop\Repository\DeliveryMethodRepository;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\IsGranted;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Annotation\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;

/**
 * @IsGranted({"ROLE_SHIPPING_METHOD","ROLE_ADMIN"})
 * Class DeliveryMethodController
 * @package CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Admin
 * @Route("/delivery-method")
 */
class DeliveryMethodController extends AbstractController
{
    private $deliveryMethodRepository;

    public function __construct(DeliveryMethodRepository $deliveryMethodRepository)
    {
        $this->deliveryMethodRepository = $deliveryMethodRepository;
    }

    /**
     * @route("", name="admin_delivery_method")
     * @Template("@CpCreationVitiCore/admin/delivery_method/index.html.twig")
     */
    public function indexAction()
    {
        $methods = $this->deliveryMethodRepository->findByPositionOrderAdmin();

        return [
            "items" => $methods,
        ];
    }


    /**
     * @param Request $request
     * @Route("/new", name="admin_delivery_method_new")
     * @Template("@CpCreationVitiCore/admin/delivery_method/new.html.twig")
     *
     * @return array|\Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function newAction(Request $request)
    {

        $form = $this->createForm(DeliveryMethodType::class);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            /** @var DeliveryMethodCategory $data */
            $data = $form->getData();
            $this->deliveryMethodRepository->save($data);
            $this->addFlash('alert-success', 'La catégorie a été ajouté avec succès');

            return $this->redirectToRoute('admin_delivery_method');

        }

        return [
            "form" => $form->createView(),
        ];
    }

    /**
     * @param DeliveryMethod $method
     * @param Request $request
     * @Route("/{id}/edit", name="admin_delivery_method_edit")
     * @Template("@CpCreationVitiCore/admin/delivery_method/edit.html.twig")
     *
     * @return array|\Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function editAction(DeliveryMethod $method, Request $request)
    {
        $form = $this->createForm(DeliveryMethodType::class, $method);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            /** @var DeliveryMethod $data */
            $data = $form->getData();
            $this->deliveryMethodRepository->save($data);
            $this->addFlash('alert-success', 'La méthode de livraison a été modifié avec succès');

            return $this->redirectToRoute('admin_delivery_method');

        }

        return [
            "item" => $method,
            "form" => $form->createView(),
        ];
    }

    /**
     * @param DeliveryMethod $method
     * @Route("/{id}/delete", name="admin_delivery_method_delete")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function deleteAction(DeliveryMethod $method)
    {
        $this->deliveryMethodRepository->softDelete($method);
        $this->addFlash('alert-success', 'La méthode de livraison a été supprimé avec succès.');

        return $this->redirectToRoute('admin_delivery_method');
    }

    /**
     * @param DeliveryMethod $method
     * @Route("/{id}/moveUp", name="admin_delivery_method_move_up")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function moveUpAction(DeliveryMethod $method)
    {
        $method->setPosition($method->getPosition() - 1);
        $this->deliveryMethodRepository->save($method);
        $this->addFlash('alert-success', 'La méthode de livraison '.$method->getName().' a été remonté.');

        return $this->redirectToRoute('admin_delivery_method');
    }

    /**
     * @param DeliveryMethod $method
     * @Route("/{id}/moveDown", name="admin_delivery_method_move_down")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function moveDownAction(DeliveryMethod $method)
    {
        $method->setPosition($method->getPosition() + 1);
        $this->deliveryMethodRepository->save($method);
        $this->addFlash('alert-success', 'La méthode de livraison '.$method->getName().' a été descendu.');

        return $this->redirectToRoute('admin_delivery_method');
    }

    /**
     * @param DeliveryMethod $method
     * @Route("/{id}/disable", name="admin_delivery_method_disable")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function disableAction(DeliveryMethod $method)
    {
        $method->setDisabled(1);
        $this->deliveryMethodRepository->save($method);
        $this->addFlash('alert-success', 'La méthode de livraison '.$method->getName().' a été dépublié.');

        return $this->redirectToRoute('admin_delivery_method');
    }

    /**
     * @param DeliveryMethod $method
     * @Route("/{id}/enable", name="admin_delivery_method_enable")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function enableAction(DeliveryMethod $method)
    {
        $method->setDisabled(0);
        $this->deliveryMethodRepository->save($method);
        $this->addFlash('alert-success', 'La méthode de livraison '.$method->getName().' a été publié.');

        return $this->redirectToRoute('admin_delivery_method');
    }
}
