<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Shop\Model;

use CpCreation\VitiCore\Behavior\HasTimestamp;
use CpCreation\VitiCore\Behavior\HasTranslatable;
use CpCreation\VitiCore\Behavior\Impl\Timestamp;
use CpCreation\VitiCore\Behavior\Impl\Translatable;
use CpCreation\VitiCore\Product\Model\ProductType;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Ramsey\Uuid\Uuid;
use Ramsey\Uuid\UuidInterface;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * Class ShopConfigMultiple
 *
 * @ORM\Table(name="shop_shop_config_multiple")
 * @ORM\HasLifecycleCallbacks()
 * @ORM\Entity(repositoryClass="CpCreation\VitiCore\Shop\Repository\ShopConfigRepository")
 */
class ShopConfigMultiple implements HasTimestamp
{
    use Timestamp;

    /**
     * @ORM\Id()
     * @ORM\Column(type="uuid")
     * @ORM\GeneratedValue(strategy="NONE")
     *
     * @var UuidInterface
     */
    private $id;

    /**
     * @ORM\Column(type="integer")
     * @Assert\NotBlank()
     *
     * @var integer|null
     */
    private $qtyOf;

    /**
     * @ORM\Column(type="integer")
     * @Assert\NotBlank()
     *
     * @var integer|null
     */
    private $qtyAt;

    /**
     * @ORM\Column(type="integer")
     * @Assert\NotBlank()
     *
     * @var integer|null
     */
    private $multiple;

    /**
     * @ORM\Column(type="boolean")
     *
     * @var boolean
     */
    private $minimum = false;

    /**
     * @ORM\ManyToMany(targetEntity="ShippingZone")
     * @Assert\Count( min = 1 , minMessage="Vous devez ajouter au moins une zone")
     *
     * @var Collection<ShippingZone>
     */
    private $zones;

    /**
     * @ORM\ManyToMany(targetEntity="CpCreation\VitiCore\Product\Model\ProductType")
     * @Assert\Count( min = 1 , minMessage="Vous devez ajouter au moins un type")
     * @var Collection<ProductType>
     */
    private $types;

    /**
     * @ORM\ManyToOne(targetEntity="CpCreation\VitiCore\Shop\Model\ShopConfig", inversedBy="multiples")
     *
     * @var ShopConfig
     */
    private $config;

    public function __construct()
    {
        $this->id    = Uuid::uuid4();
        $this->zones = new ArrayCollection();
        $this->types = new ArrayCollection();
    }

    /**
     * @return UuidInterface
     */
    public function getId(): UuidInterface
    {
        return $this->id;
    }

    /**
     * @return int|null
     */
    public function getQtyOf(): ?int
    {
        return $this->qtyOf;
    }

    /**
     * @param int|null $qtyOf
     */
    public function setQtyOf(?int $qtyOf): void
    {
        $this->qtyOf = $qtyOf;
    }

    /**
     * @return int|null
     */
    public function getQtyAt(): ?int
    {
        return $this->qtyAt;
    }

    /**
     * @param int|null $qtyAt
     */
    public function setQtyAt(?int $qtyAt): void
    {
        $this->qtyAt = $qtyAt;
    }

    /**
     * @return int|null
     */
    public function getMultiple(): ?int
    {
        return $this->multiple;
    }

    /**
     * @param int|null $multiple
     */
    public function setMultiple(?int $multiple): void
    {
        $this->multiple = $multiple;
    }

    /**
     * @return ShopConfig
     */
    public function getConfig(): ShopConfig
    {
        return $this->config;
    }

    /**
     * @param ShopConfig $config |null
     */
    public function setConfig(?ShopConfig $config): void
    {
        $this->config = $config;
    }

    /**
     * @param ProductType $productType
     * @return $this
     */
    public function addType(ProductType $productType)
    {
        $this->types[] = $productType;

        return $this;
    }

    /**
     * @param ProductType $productType
     */
    public function removeType(ProductType $productType)
    {
        $this->types->removeElement($productType);
    }

    /**
     * @return Collection<ProductType>
     */
    public function getTypes(): Collection
    {
        return $this->types;
    }

    /**
     * @return bool
     */
    public function isMinimum(): bool
    {
        return $this->minimum;
    }

    /**
     * @param bool $minimum
     */
    public function setMinimum(bool $minimum): void
    {
        $this->minimum = $minimum;
    }
    
    /**
     * @param ShippingZone $zone
     * @return $this
     */
    public function addZone(ShippingZone $zone)
    {
        $this->zones[] = $zone;

        return $this;
    }

    /**
     * @param ShippingZone $zone
     */
    public function removeZone(ShippingZone $zone)
    {
        $this->zones->removeElement($zone);
    }

    /**
     * @return Collection<ShippingZone>
     */
    public function getZones(): Collection
    {
        return $this->zones;
    }
}
