<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Shop\Model;

use CpCreation\VitiCore\Behavior\HasTimestamp;
use CpCreation\VitiCore\Behavior\HasTranslatable;
use CpCreation\VitiCore\Behavior\Impl\Timestamp;
use CpCreation\VitiCore\Behavior\Impl\Translatable;
use CpCreation\VitiCore\Event\Model\EventImage;
use CpCreation\VitiCore\Product\Model\ProductPriceGroup;
use DateInterval;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Ramsey\Uuid\Uuid;
use Ramsey\Uuid\UuidInterface;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * Class ShopConfig
 *
 * @ORM\Table(name="shop_shop_config")
 * @ORM\HasLifecycleCallbacks()
 * @ORM\Entity(repositoryClass="CpCreation\VitiCore\Shop\Repository\ShopConfigRepository")
 */
class ShopConfig implements HasTimestamp, HasTranslatable
{
    use Translatable, Timestamp;

  /**
     * @Assert\Valid
     * @AcmeAssert\Translation
     * @ORM\OneToMany(targetEntity="CpCreation\VitiCore\Shop\Model\Translation\ShopConfigTranslation", mappedBy="translatable", cascade={"persist","remove","merge"}, indexBy="locale", orphanRemoval="true")
     *
     * @var TranslationInterface[]|Collection
     */
    protected $translations;
    
    /**
     * @ORM\Id()
     * @ORM\Column(type="uuid")
     * @ORM\GeneratedValue(strategy="NONE")
     *
     * @var UuidInterface
     */
    private $id;

    /**
     * @ORM\Column(type="boolean")
     *
     * @var boolean
     */
    private $viewCategory;

    /**
     * @ORM\Column(type="boolean")
     *
     * @var boolean
     */
    private $viewOrderByCategory;

    /**
     * @ORM\Column(type="boolean")
     *
     * @var boolean
     */
    private $hidePriceDefault;

    /**
     * @ORM\Column(type="boolean")
     *
     * @var boolean
     */
    private $enableAwardShop;

    /**
     * @ORM\Column(type="boolean")
     *
     * @var boolean
     */
    private $viewFavoriteCat;

    /**
     * @ORM\Column(type="boolean")
     *
     * @var boolean
     */
    private $disableLogoBank;

    /**
     * @Assert\Type(type="integer")
     *
     * @ORM\Column(type="integer", nullable=true)
     *
     * @var int|null
     */
    private $maxAwardsVisibleInShop;

    /**
     * @ORM\Column(type="boolean")
     *
     * @var boolean
     */
    private $enabledCityField;

    /**
     * @ORM\OneToMany(targetEntity="CpCreation\VitiCore\Shop\Model\ShopConfigDisableCountry", cascade={"all"}, mappedBy="config", orphanRemoval=true)
     * @ORM\OrderBy({"country" = "ASC"})
     * @Assert\Valid()
     *
     * @var Collection|ShopConfigDisableCountry[]
     */
    private $disableCountries;

    /**
     * @ORM\Column(type="json", nullable=true, options={"default" : "[]"})
     *
     * @var array|null
     */
    private $enableCountriesForShop;

    /**
     * @var \DateTime|null
     * @ORM\Column(type="datetime", nullable=true)
     */
    private $vacationOf;

    /**
     * @var \DateTime|null
     * @ORM\Column(type="datetime", nullable=true)
     */
    private $vacationAt;

    /**
     * @ORM\Column(type="boolean")
     *
     * @var boolean
     */
    private $disableShopForVacation;

    /**
     * @ORM\OneToMany(targetEntity="CpCreation\VitiCore\Shop\Model\ShopConfigMultiple", cascade={"all"}, mappedBy="config", orphanRemoval=true)
     * @ORM\OrderBy({"qtyOf" = "ASC"})
     * @Assert\Valid()
     *
     * @var Collection|ShopConfigMultiple[]
     */
    private $multiples;

    /**
     * @ORM\Column(type="string",nullable=true)
     *
     * @var string | null
     */
    private $giftCardCheckToken = null;

    /**
     * @ORM\Column(type="dateinterval",nullable=true)
     *
     * @var DateInterval | null
     */
    private $giftCardValidTime = null;

    public function __construct()
    {
        $this->id                     = Uuid::uuid4();
        $this->viewCategory           = 1;
        $this->hidePriceDefault       = 0;
        $this->enableAwardShop        = false;
        $this->disableLogoBank        = false;
        $this->disableCountries       = [];
        $this->enableCountriesForShop = [];
        $this->viewOrderByCategory    = false;
        $this->viewFavoriteCat        = false;
        $this->enabledCityField       = false;
        $this->disableShopForVacation = false;
        $this->multiples              = new ArrayCollection();
    }

    /**
     * @return UuidInterface
     */
    public function getId(): UuidInterface
    {
        return $this->id;
    }

    /**
     * @return bool
     */
    public function isViewCategory(): bool
    {
        return $this->viewCategory;
    }

    /**
     * @param bool $viewCategory
     */
    public function setViewCategory(bool $viewCategory): void
    {
        $this->viewCategory = $viewCategory;
    }

    /**
     * @return bool
     */
    public function isViewOrderByCategory(): bool
    {
        return $this->viewOrderByCategory;
    }

    /**
     * @param bool $viewOrderByCategory
     */
    public function setViewOrderByCategory(bool $viewOrderByCategory): void
    {
        $this->viewOrderByCategory = $viewOrderByCategory;
    }

    /**
     * @return bool
     */
    public function isDisableLogoBank(): bool
    {
        return $this->disableLogoBank;
    }

    /**
     * @param bool $disableLogoBank
     */
    public function setDisableLogoBank(bool $disableLogoBank): void
    {
        $this->disableLogoBank = $disableLogoBank;
    }

    /**
     * @return bool
     */
    public function isHidePriceDefault(): bool
    {
        return $this->hidePriceDefault;
    }

    /**
     * @param bool $hidePriceDefault
     */
    public function setHidePriceDefault(bool $hidePriceDefault): void
    {
        $this->hidePriceDefault = $hidePriceDefault;
    }

    /**
     * @return bool
     */
    public function isEnableAwardShop(): bool
    {
        return $this->enableAwardShop;
    }

    /**
     * @param bool $enableAwardShop
     */
    public function setEnableAwardShop(bool $enableAwardShop): void
    {
        $this->enableAwardShop = $enableAwardShop;
    }

    /**
     * @return bool
     */
    public function isViewFavoriteCat(): bool
    {
        return $this->viewFavoriteCat;
    }

    /**
     * @param bool $viewFavoriteCat
     */
    public function setViewFavoriteCat(bool $viewFavoriteCat): void
    {
        $this->viewFavoriteCat = $viewFavoriteCat;
    }

    /**
     * @param ShopConfigDisableCountry $disableCountry
     */
    public function addDisableCountry(ShopConfigDisableCountry $disableCountry)
    {
        if ($this->disableCountries->contains($disableCountry)) {

            return;
        }

        $disableCountry->setConfig($this);
        $this->disableCountries->add($disableCountry);
    }

    /**
     * @param ShopConfigDisableCountry $disableCountry
     */
    public function removeDisableCountry(ShopConfigDisableCountry $disableCountry)
    {
        if (!$this->disableCountries->contains($disableCountry)) {
            return;
        }
        $this->disableCountries->removeElement($disableCountry);
    }

    /**
     * @return ShopConfigDisableCountry[]|Collection
     */
    public function getDisableCountries(): Collection
    {
        return $this->disableCountries;
    }

    /**
     * @return array|null
     */
    public function getEnableCountriesForShop(): ?array
    {
        return $this->enableCountriesForShop;
    }

    /**
     * @param array|null $enableCountriesForShop
     */
    public function setEnableCountriesForShop(?array $enableCountriesForShop): void
    {
        $this->enableCountriesForShop = $enableCountriesForShop;
    }

    /**
     * @return int|null
     */
    public function getMaxAwardsVisibleInShop(): ?int
    {
        return $this->maxAwardsVisibleInShop;
    }

    /**
     * @param int|null $maxAwardsVisibleInShop
     */
    public function setMaxAwardsVisibleInShop(?int $maxAwardsVisibleInShop): void
    {
        $this->maxAwardsVisibleInShop = $maxAwardsVisibleInShop;
    }

    /**
     * @return bool
     */
    public function isEnabledCityField(): bool
    {
        return $this->enabledCityField;
    }

    /**
     * @param bool $enabledCityField
     */
    public function setEnabledCityField(bool $enabledCityField): void
    {
        $this->enabledCityField = $enabledCityField;
    }

    /**
     * @param ShopConfigMultiple $multiple
     */
    public function addMultiple(ShopConfigMultiple $multiple)
    {
        if ($this->multiples->contains($multiple)) {

            return;
        }

        $multiple->setConfig($this);
        $this->multiples->add($multiple);
    }

    /**
     * @param ShopConfigMultiple $multiple
     */
    public function removeMultiple(ShopConfigMultiple $multiple)
    {
        if (!$this->multiples->contains($multiple)) {
            return;
        }

        $multiple->setConfig(null);
        $this->multiples->removeElement($multiple);
    }

    /**
     * @return \DateTime|null
     */
    public function getVacationOf(): ?\DateTime
    {
        return $this->vacationOf;
    }

    /**
     * @param \DateTime|null $vacationOf
     */
    public function setVacationOf(?\DateTime $vacationOf): void
    {
        $this->vacationOf = $vacationOf;
    }

    /**
     * @return \DateTime|null
     */
    public function getVacationAt(): ?\DateTime
    {
        return $this->vacationAt;
    }

    /**
     * @param \DateTime|null $vacationAt
     */
    public function setVacationAt(?\DateTime $vacationAt): void
    {
        $this->vacationAt = $vacationAt;
    }

    /**
     * @return bool
     */
    public function isDisableShopForVacation(): bool
    {
        return $this->disableShopForVacation;
    }

    /**
     * @param bool $disableShopForVacation
     */
    public function setDisableShopForVacation(bool $disableShopForVacation): void
    {
        $this->disableShopForVacation = $disableShopForVacation;
    }

    /**
     * @return ShopConfigMultiple[]|Collection
     */
    public function getMultiples(): Collection
    {
        return $this->multiples;
    }

    /**
     * @return null|string
     */
    public function getGiftCardCheckToken(): ?string
    {
        return $this->giftCardCheckToken;
    }

    /**
     * @param null|string $giftCardCheckToken
     */
    public function setGiftCardCheckToken(?string $giftCardCheckToken): void
    {
        $this->giftCardCheckToken = $giftCardCheckToken;
    }

    /**
     * @return null|DateInterval
     */
    public function getGiftCardValidTime(): ?DateInterval
    {
        return $this->giftCardValidTime;
    }

    /**
     * @param null|DateInterval $giftCardValidTime
     */
    public function setGiftCardValidTime(?DateInterval $giftCardValidTime): void
    {
        $this->giftCardValidTime = $giftCardValidTime;
    }
}
