<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Shop\Model;


use CpCreation\VitiCore\Behavior\HasTimestamp;
use CpCreation\VitiCore\Behavior\Impl\Timestamp;
use CpCreation\VitiCore\Product\Model\ProductPriceGroup;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Ramsey\Uuid\Uuid;
use Ramsey\Uuid\UuidInterface;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * Class ShippingAdditional
 *
 * @ORM\Entity(repositoryClass="CpCreation\VitiCore\Shop\Repository\ShippingAdditionalRepository")
 * @ORM\Table(name="shop_shipping_additional")
 * @ORM\HasLifecycleCallbacks()
 */
class ShippingAdditional implements HasTimestamp
{
    use Timestamp;

    /**
     * @ORM\Id()
     * @ORM\Column(type="uuid")
     * @ORM\GeneratedValue(strategy="NONE")
     *
     * @var UuidInterface
     */
    private $id;

    /**
     * @ORM\Column(type="integer")
     * @Assert\NotBlank()
     *
     * @var integer
     */
    private $qtyOf;

    /**
     * @ORM\Column(type="integer")
     * @Assert\NotBlank()
     *
     * @var integer
     */
    private $qtyAt;

    /**
     * @ORM\Column(type="integer")
     *
     * @var integer
     */
    private $dateOf;

    /**
     * @ORM\Column(type="integer")
     *
     * @var integer
     */
    private $dateAt;

    /**
     * @ORM\Column(type="integer")
     *
     * @var int
     */
    private $price;

    /**
     * @ORM\ManyToMany(targetEntity="ShippingCounty")
     * @ORM\JoinTable(name="shop_shipping_additional_counties",
     *      joinColumns={@ORM\JoinColumn(name="shipping_additional_id", referencedColumnName="id")},
     *      inverseJoinColumns={@ORM\JoinColumn(name="countie_id", referencedColumnName="id")})
     * @ORM\OrderBy({"code" = "ASC"})
     * @Assert\Valid()
     *
     * @var Collection<ShippingCounty>
     */
    private $counties;

    /**
     * @ORM\ManyToMany(targetEntity="CpCreation\VitiCore\Product\Model\ProductPriceGroup")
     * @Assert\Count( min = 1 , minMessage="Vous devez ajouter au moins un groupe")
     * @var Collection<ProductPriceGroup>
     */
    private $groups;

    public function __construct()
    {
        $this->id       = Uuid::uuid4();
        $this->counties = new ArrayCollection();
        $this->groups   = new ArrayCollection();
    }

    /**
     * @return UuidInterface
     */
    public function getId(): UuidInterface
    {
        return $this->id;
    }

    /**
     * @return int
     */
    public function getQtyOf(): ?int
    {
        return $this->qtyOf;
    }

    /**
     * @param int $qtyOf
     */
    public function setQtyOf(int $qtyOf = null)
    {
        $this->qtyOf = $qtyOf;
    }

    /**
     * @return int
     */
    public function getQtyAt(): ?int
    {
        return $this->qtyAt;
    }

    /**
     * @param int $qtyAt
     */
    public function setQtyAt(int $qtyAt = null)
    {
        $this->qtyAt = $qtyAt;
    }

    /**
     * @return int
     */
    public function getDateOf(): ?int
    {
        return $this->dateOf;
    }

    /**
     * @param int $dateOf
     */
    public function setDateOf(int $dateOf = null)
    {
        $this->dateOf = $dateOf;
        //Correspond au 31 décembre
        if (!$dateOf) {
            $this->$dateOf = 1231;
        }
    }

    /**
     * @return int
     */
    public function getDateAt(): ?int
    {
        return $this->dateAt;
    }

    /**
     * @param int $dateAt
     */
    public function setDateAt(int $dateAt = null)
    {
        $this->dateAt = $dateAt;
        //Correspond au premier janvier
        if (!$dateAt) {
            $this->dateAt = 101;
        }
    }

    /**
     * @return int
     */
    public function getPrice(): ?int
    {
        return $this->price;
    }

    /**
     * @param int $price
     */
    public function setPrice(int $price)
    {
        $this->price = $price;
    }

    /**
     * @param ShippingCounty $county
     * @return $this
     */
    public function addCounty(ShippingCounty $county)
    {
        $this->counties[] = $county;

        return $this;
    }

    /**
     * @param ShippingCounty $county
     */
    public function removeCounty(ShippingCounty $county)
    {
        $this->counties->removeElement($county);
    }

    /**
     * @return Collection<ShippingCounty>
     */
    public function getCounties(): Collection
    {
        return $this->counties;
    }

    /**
     * @param ProductPriceGroup $group
     * @return $this
     */
    public function addGroup(ProductPriceGroup $group)
    {
        $this->groups[] = $group;

        return $this;
    }

    /**
     * @param ProductPriceGroup $group
     */
    public function removeGroup(ProductPriceGroup $group)
    {
        $this->groups->removeElement($group);
    }

    /**
     * @return Collection<ProductPriceGroup>
     */
    public function getGroups(): Collection
    {
        return $this->groups;
    }

}
