<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Admin;

use CpCreation\VitiCore\Bridge\Symfony\Bundle\Form\Admin\PromoType;
use CpCreation\VitiCore\Cart\Model\Promo;
use CpCreation\VitiCore\Cart\Repository\PromoRepository;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Security;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\Routing\Annotation\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;
use Symfony\Component\HttpFoundation\Request;

/**
 * @Security("is_granted('ROLE_ADMIN') or is_granted('ROLE_PROMO')")
 * Class PromoController
 * @package CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Admin
 * @Route("/code-promo")
 */
class PromoController extends AbstractController
{
    private $promoRepository;

    public function __construct(PromoRepository $promoRepository)
    {
        $this->promoRepository =  $promoRepository;
    }

    /**
     * @route("", name="admin_promo")
     * @Template("@CpCreationVitiCore/admin/promo/index.html.twig")
     */
    public function indexAction()
    {
        $promos = $this->promoRepository->findBy([], ['endAt' => 'DESC']);

        return [
            "items" => $promos
        ];
    }

    /**
     * @param Request $request
     * @Route("/new", name="admin_promo_new")
     * @Template("@CpCreationVitiCore/admin/promo/new.html.twig")
     *
     * @return array|\Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function newAction(Request $request)
    {

        $form = $this->createForm(PromoType::class);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            /** @var Promo $data */
            $data = $form->getData();
            $this->promoRepository->save($data);
            $this->addFlash('alert-success', 'Le code remise '.$data->getTitle().' a été ajouté avec succès');

            return $this->redirectToRoute('admin_promo');

        }

        return [
            "form" => $form->createView(),
        ];
    }

    /**
     * @param Promo $promo
     * @param Request $request
     * @Route("/{id}/edit", name="admin_promo_edit")
     * @Template("@CpCreationVitiCore/admin/promo/edit.html.twig")
     *
     * @return array|\Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function editAction(Promo $promo, Request $request)
    {
        $form = $this->createForm(PromoType::class, $promo);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            /** @var Promo $data */
            $data = $form->getData();
            $this->promoRepository->save($data);
            $this->addFlash('alert-success', 'Le code remise '.$promo->getTitle().' a été modifié avec succès');

            return $this->redirectToRoute('admin_promo');

        }

        return [
            "item" => $promo,
            "form" => $form->createView(),
        ];
    }

    /**
     * @param Promo $promo
     * @Route("/{id}/delete", name="admin_promo_delete")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function deleteAction(Promo $promo)
    {
        $this->promoRepository->delete($promo);
        $this->addFlash('alert-success', 'Le code remise '.$promo->getTitle().' a été supprimé avec succès.');

        return $this->redirectToRoute('admin_promo');
    }


    /**
     * @param Promo $promo
     * @Route("/{id}/disable", name="admin_promo_disable")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function disableAction(Promo $promo)
    {
        $promo->setDisabled(1);
        $this->promoRepository->save($promo);
        $this->addFlash('alert-success', 'Le code remise '.$promo->getTitle().' a été dépublié.');

        return $this->redirectToRoute('admin_promo');
    }

    /**
     * @param Promo $promo
     * @Route("/{id}/enable", name="admin_promo_enable")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function enableAction(Promo $promo)
    {
        $promo->setDisabled(0);
        $this->promoRepository->save($promo);
        $this->addFlash('alert-success', 'Le code remise '.$promo->getTitle().' a été publié.');

        return $this->redirectToRoute('admin_promo');
    }
}
