<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Gite\Model;

use CpCreation\VitiCore\Behavior\HasFile;
use CpCreation\VitiCore\Behavior\HasImage;
use CpCreation\VitiCore\Behavior\HasTimestamp;
use CpCreation\VitiCore\Behavior\HasTranslatable;
use CpCreation\VitiCore\Behavior\HasVideo;
use CpCreation\VitiCore\Behavior\Impl\File;
use CpCreation\VitiCore\Behavior\Impl\Image;
use CpCreation\VitiCore\Behavior\Impl\Timestamp;
use CpCreation\VitiCore\Behavior\Impl\Translatable;
use CpCreation\VitiCore\Behavior\Impl\Video;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Ramsey\Uuid\Uuid;
use Ramsey\Uuid\UuidInterface;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * Class Gite
 *
 * @ORM\Entity(repositoryClass="CpCreation\VitiCore\Gite\Repository\GiteRepository")
 * @ORM\Table(name="gite_gite")
 * @ORM\HasLifecycleCallbacks()
 */
class Gite implements HasTimestamp, HasVideo, HasFile, HasImage, HasTranslatable
{
    use Timestamp, Video, Image, File, Translatable;

    /**
     * @ORM\Id()
     * @ORM\Column(type="uuid")
     * @ORM\GeneratedValue(strategy="NONE")
     *
     * @var UuidInterface
     */
    private $id;

    /**
     * @Gedmo\SortablePosition()
     * @ORM\Column(type="integer", nullable=true)
     *
     * @var int
     */
    private $position;

    /**
     * @ORM\OneToMany(targetEntity="Room", cascade={"persist","remove"}, mappedBy="gite", orphanRemoval=true)
     * @ORM\OrderBy({"position"="ASC"})
     * @Assert\Valid()
     *
     * @var Collection<Room>
     */
    private $rooms;

    /**
     * @ORM\OneToMany(targetEntity="CpCreation\VitiCore\Gite\Model\GiteImage", cascade={"persist","remove"}, mappedBy="gite")
     * @ORM\OrderBy({"position" = "ASC"})
     * @Assert\Valid()
     *
     * @var Collection<GiteImage>
     */
    private $images;

    public function __construct()
    {
        $this->id = Uuid::uuid4();
        $this->rooms = new ArrayCollection();
        $this->images = new ArrayCollection();
    }

    /**
     * @return UuidInterface
     */
    public function getId(): UuidInterface
    {
        return $this->id;
    }

    /**
     * @return int
     */
    public function getPosition(): ?int
    {
        return $this->position;
    }

    /**
     * @param int $position
     */
    public function setPosition(int $position)
    {
        $this->position = $position;
    }

    /**
     * @param Room $room
     * @return $this
     */
    public function addRoom(Room $room)
    {
        $room->setGite($this);
        $this->rooms[] = $room;

        return $this;
    }

    /**
     * @param Room $room
     */
    public function removeRoom(Room $room)
    {
        $this->rooms->removeElement($room);
        $room->setGite(null);
    }
    
    /**
     * @return Collection
     */
    public function getRooms(): Collection
    {
        return $this->rooms;
    }

    public function addImage(GiteImage $image): self
    {

        if (!$this->images->contains($image)) {
            $this->images[] = $image;
            $image->setGite($this);
        }

        return $this;
    }

    public function removeImage(GiteImage $image): self
    {

        if ($this->images->contains($image)) {
            $this->images->removeElement($image);
            // set the owning side to null (unless already changed)
            if ($image->getGite() === $this) {
                $image->setGite(null);
            }
        }

        return $this;
    }

    /**
     * @return Collection<Image>
     */
    public function getImages(): Collection
    {
        return $this->images;
    }
    
}