<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Event\Model;

use CpCreation\VitiCore\Behavior\HasDisable;
use CpCreation\VitiCore\Behavior\HasFile;
use CpCreation\VitiCore\Behavior\HasGeocoder;
use CpCreation\VitiCore\Behavior\HasTimestamp;
use CpCreation\VitiCore\Behavior\HasTranslatable;
use CpCreation\VitiCore\Behavior\Impl\Disable;
use CpCreation\VitiCore\Behavior\Impl\Geocodable;
use CpCreation\VitiCore\Behavior\Impl\Timestamp;
use CpCreation\VitiCore\Behavior\Impl\File;
use CpCreation\VitiCore\Behavior\Impl\Translatable;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Ramsey\Uuid\Uuid;
use Ramsey\Uuid\UuidInterface;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * Class Event
 *
 * @ORM\Entity(repositoryClass="CpCreation\VitiCore\Event\Repository\EventRepository")
 * @ORM\Table(name="event_event")
 * @ORM\HasLifecycleCallbacks()
 */

class Event implements HasTimestamp,HasDisable, HasFile, HasTranslatable, HasGeocoder
{
    use Timestamp, File, Disable, Translatable,Geocodable;

    /**
     * @ORM\Id()
     * @ORM\Column(type="uuid")
     * @ORM\GeneratedValue(strategy="NONE")
     *
     * @var UuidInterface
     */
    private $id;

    /**
     * @ORM\Column(type="datetime")
     * @Assert\NotBlank()
     * @var \DateTimeInterface
     */
    private $beginAt;

    /**
     * @ORM\Column(type="datetime", nullable=true)
     *
     * @var \DateTimeInterface
     */
    private $endAt;

    /**
     * @ORM\Column(type="text", nullable=true)
     *
     * @var string
     */
    private $url;

    /**
     * @ORM\OneToMany(targetEntity="CpCreation\VitiCore\Event\Model\EventImage", cascade={"persist","remove"}, mappedBy="event")
     * @ORM\OrderBy({"position" = "ASC"})
     * @Assert\Valid()
     *
     * @var Collection<EventImage>
     */
    private $images;

    public function __construct()
    {
        $this->id = Uuid::uuid4();
        $this->images   = new ArrayCollection();
        $this->disabled = false;
    }

    /**
     * @return UuidInterface
     */
    public function getId(): UuidInterface
    {
        return $this->id;
    }

    /**
     * @return \DateTimeInterface beginAt
     */
    public function getBeginAt(): ?\DateTime
    {
        return $this->beginAt;
    }

    /**
     * @param \DateTimeInterface $beginAt
     */
    public function setBeginAt(?\DateTimeInterface $beginAt)
    {
        $this->beginAt = $beginAt;
    }

    /**
     * @return \DateTimeInterface endAt
     */
    public function getEndAt(): ?\DateTime
    {
        return $this->endAt;
    }

    /**
     * @param \DateTimeInterface|null $endAt
     */
    public function setEndAt(\DateTimeInterface $endAt = null)
    {
        $this->endAt = $endAt;
    }

    /**
     * @return string
     */
    public function getUrl(): ?string
    {
        return $this->url;
    }

    /**
     * @param string $url
     */
    public function setUrl(string $url = null)
    {
        $this->url = $url;
    }

    /**
     * @param EventImage $image
     * @return $this
     */
    public function addImage(EventImage $image)
    {

        if (!$this->images->contains($image)) {
            $this->images[] = $image;
            $image->setEvent($this);
        }

        return $this;
    }

    /**
     * @param EventImage $image
     * @return $this
     */
    public function removeImage(EventImage $image)
    {

        if ($this->images->contains($image)) {
            $this->images->removeElement($image);
            // set the owning side to null (unless already changed)
            if ($image->getEvent() === $this) {
                $image->setEvent(null);
            }
        }

        return $this;
    }

    /**
     * @return Collection<Image>
     */
    public function getImages(): Collection
    {
        return $this->images;
    }
}
