<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Admin;

use CpCreation\VitiCore\Bridge\Symfony\Bundle\Form\Admin\GiteType;
use CpCreation\VitiCore\Gite\Model\Gite;
use CpCreation\VitiCore\Gite\Repository\GiteRepository;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Security;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\Routing\Annotation\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;
use Symfony\Component\HttpFoundation\Request;

/**
 * @Security("is_granted('ROLE_ADMIN')")
 * Class GiteController
 * @package CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Admin
 * @Route("/gite")
 */
class GiteController extends AbstractController
{
    private $giteRepository;

    public function __construct(GiteRepository $giteRepository)
    {
        $this->giteRepository = $giteRepository;
    }

    /**
     * @route("", name="admin_gite")
     * @Template("@CpCreationVitiCore/admin/gite/index.html.twig")
     */
    public function indexAction()
    {
        $gites = $this->giteRepository->findBy([], ['position' => 'ASC']);

        return [
            "items" => $gites,
        ];
    }

    /**
     * @param Request $request
     * @Route("/new", name="admin_gite_new")
     * @Template("@CpCreationVitiCore/admin/gite/new.html.twig")
     *
     * @return array|\Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function newAction(Request $request)
    {

        $form = $this->createForm(GiteType::class);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            /** @var Gite $data */
            $data = $form->getData();
            $this->giteRepository->save($data);
            $this->addFlash('alert-success', 'Le gite '.$data->getTitle().' a été ajouté avec succès');

            return $this->redirectToRoute('admin_gite');
        }

        return [
            "form" => $form->createView(),
        ];
    }

    /**
     * @param Gite    $gite
     * @param Request $request
     * @Route("/{id}/edit", name="admin_gite_edit")
     * @Template("@CpCreationVitiCore/admin/gite/edit.html.twig")
     *
     * @return array|\Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function editAction(Gite $gite, Request $request)
    {
        $form = $this->createForm(GiteType::class, $gite);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            /** @var Gite $data */
            $data = $form->getData();
            $data->setUpdatedAt(new \DateTime()); // Forcer Update de doctrine
            $this->giteRepository->save($data);
            $this->addFlash('alert-success', 'Le gite '.$gite->getTitle().' a été modifié avec succès');

            return $this->redirectToRoute('admin_gite');
        }

        return [
            "item" => $gite,
            "form" => $form->createView(),
        ];
    }

    /**
     * @param Gite $gite
     * @Route("/{id}/moveUp", name="admin_gite_move_up")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function moveUpAction(Gite $gite)
    {
        $gite->setPosition($gite->getPosition() - 1);
        $this->giteRepository->save($gite);
        $this->addFlash('alert-success', 'La locale '.$gite->getTitle().' a été remonté.');

        return $this->redirectToRoute('admin_gite');
    }

    /**
     * @param Gite $gite
     * @Route("/{id}/moveDown", name="admin_gite_move_down")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function moveDownAction(Gite $gite)
    {
        $gite->setPosition($gite->getPosition() + 1);
        $this->giteRepository->save($gite);
        $this->addFlash('alert-success', 'La locale '.$gite->getTitle().' a été descendu.');

        return $this->redirectToRoute('admin_gite');
    }

    /**
     * @param Gite $gite
     * @Route("/{id}/delete", name="admin_gite_delete")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function deleteAction(Gite $gite)
    {
        $this->giteRepository->delete($gite);
        $this->addFlash('alert-success', 'La locale '.$gite->getTitle().' a été supprimé avec succès.');

        return $this->redirectToRoute('admin_gite');
    }
}
