<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Admin;

use CpCreation\VitiCore\Bridge\Symfony\Bundle\Form\Admin\ShippingCountryType;
use CpCreation\VitiCore\Shop\Model\ShippingCountry;
use CpCreation\VitiCore\Shop\Repository\ShippingCountryRepository;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Security;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Annotation\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;

/**
 * @Security("is_granted('ROLE_ADMIN')")
 * Class ShippingCountryController
 * @package CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Admin
 * @Route("/shipping-countries")
 */
class ShippingCountryController extends AbstractController
{
    private $shippingCountryRepository;

    public function __construct(ShippingCountryRepository $shippingCountryRepository)
    {
        $this->shippingCountryRepository =  $shippingCountryRepository;
    }

    /**
     * @route("", name="admin_shipping_country")
     * @Template("@CpCreationVitiCore/admin/shipping/country/index.html.twig")
     */
    public function indexAction()
    {
        $countries = $this->shippingCountryRepository->findCountriesOrder();

        return [
            "items" => $countries,
        ];
    }

    /**
     * @param Request $request
     * @Route("/new", name="admin_shipping_country_new")
     * @Template("@CpCreationVitiCore/admin/shipping/country/new.html.twig")
     *
     * @return array|\Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function newAction(Request $request)
    {
        $form = $this->createForm(ShippingCountryType::class);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            /** @var ShippingCountry $data */
            $data = $form->getData();
            $this->shippingCountryRepository->save($data);
            $this->addFlash('alert-success', 'Le pays '.$data->getName().' a été ajouté avec succès');

            return $this->redirectToRoute('admin_shipping_country');

        }

        return [
            "form" => $form->createView(),
        ];
    }

    /**
     * @param ShippingCountry $country
     * @param Request      $request
     * @Route("/{id}/edit", name="admin_shipping_country_edit")
     * @Template("@CpCreationVitiCore/admin/shipping/country/edit.html.twig")
     *
     * @return array|\Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function editAction(ShippingCountry $country, Request $request)
    {
        $form = $this->createForm(ShippingCountryType::class, $country);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            /** @var ShippingCountry $data */
            $data = $form->getData();
            $this->shippingCountryRepository->save($data);
            $this->addFlash('alert-success', 'Le pays '.$country->getName().' a été modifié avec succès');

            return $this->redirectToRoute('admin_shipping_country');

        }

        return [
            "item" => $country,
            "form" => $form->createView(),
        ];
    }

    /**
     * @param ShippingCountry $country
     * @Route("/{id}/delete", name="admin_shipping_country_delete")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function deleteAction(ShippingCountry $country)
    {
        $this->shippingCountryRepository->delete($country);
        $this->addFlash('alert-success', 'Le pays '.$country->getName().' a été supprimé avec succès.');

        return $this->redirectToRoute('admin_shipping_country');
    }

    /**
     * @param ShippingCountry $country
     * @Route("/{id}/disable", name="admin_shipping_country_disable")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function disableAction(ShippingCountry $country)
    {
        $country->setDisabled(1);
        $this->shippingCountryRepository->save($country);
        $this->addFlash('alert-success', 'Le pays '.$country->getName().' a été dépublié.');

        return $this->redirectToRoute('admin_shipping_country');
    }

    /**
     * @param ShippingCountry $country
     * @Route("/{id}/enable", name="admin_shipping_country_enable")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function enableAction(ShippingCountry $country)
    {
        $country->setDisabled(0);
        $this->shippingCountryRepository->save($country);
        $this->addFlash('alert-success', 'Le pays '.$country->getName().' a été publié.');

        return $this->redirectToRoute('admin_shipping_country');
    }


}
