<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Bridge\Doctrine\EventSubscriber;

use CpCreation\VitiCore\Bridge\Symfony\Bundle\Utils\Mailer;
use CpCreation\VitiCore\Cart\Model\CartCheckout;
use CpCreation\VitiCore\Cart\Model\CartItem;
use CpCreation\VitiCore\Cart\Model\Promo;
use CpCreation\VitiCore\Contact\Model\Contact;
use CpCreation\VitiCore\Product\Model\ProductGift;
use CpCreation\VitiCore\Shop\Model\ShopConfig;
use CpCreation\VitiCore\Shop\Repository\ShopConfigRepository;
use DateTime;
use DateTimeImmutable;
use Doctrine\Common\EventSubscriber;
use Doctrine\ORM\Event\LifecycleEventArgs;
use Doctrine\ORM\Event\OnFlushEventArgs;
use Doctrine\ORM\Events;
use Symfony\Component\HttpFoundation\RequestStack;

class CartCheckoutSubscriber implements EventSubscriber
{
    private $request;
    private $mailer;
    private $shopConfigRepository;

    /**
     * CartCheckoutSubscriber constructor.
     * @param RequestStack $request
     * @param Mailer       $mailer
     */
    public function __construct(RequestStack $request, Mailer $mailer, ShopConfigRepository $shopConfigRepository)
    {
        $this->request              = $request;
        $this->mailer               = $mailer;
        $this->shopConfigRepository = $shopConfigRepository;
    }

    public function getSubscribedEvents()
    {

        return [
            Events::prePersist,
            Events::postUpdate,
            Events::onFlush,
        ];
    }

    /**
     * @param LifecycleEventArgs $event
     */
    public function prePersist(LifecycleEventArgs $event)
    {
        $entity = $event->getObject();
        if ($entity instanceof CartCheckout) {
            $entity->setLocale($this->request->getCurrentRequest()->getLocale());
            $entity->setIpClient($this->request->getMasterRequest()->getClientIp());
        }
    }

    /**
     * @param LifecycleEventArgs $event
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException@
     * @throws \Twig\Error\LoaderError
     * @throws \Twig\Error\RuntimeError
     * @throws \Twig\Error\SyntaxError
     */
    public function postUpdate(LifecycleEventArgs $event)
    {
        $entity = $event->getObject();
        $em     = $event->getEntityManager();
        if ($entity instanceof CartCheckout) {
            if ($entity->getPayedAt()) {

                $promoText = $entity->getCart()->getPromo();
                /** @var Promo $promo */
                $promo = $em->getRepository(Promo::class)->findOneBy(['title' => $promoText]);

                if ($promo) {
                    $promo->setCount($promo->getCount() + 1);
                    $em->persist($promo);
                    $em->flush();
                }
            }
            if ($entity->isDelivery() && !$entity->getDeletedAt() && !$entity->isDisabled()) {
                $contact = $em->getRepository(Contact::class)->findOne();
                $this->mailer->senDelivery($contact, $entity->getCart());
            }
        }
    }


    public function onFlush(OnFlushEventArgs $eventArgs)
    {

        $em  = $eventArgs->getEntityManager();
        $uow = $em->getUnitOfWork();
        $uow->computeChangeSets();
        $hasGift = false;
        foreach ($uow->getScheduledEntityUpdates() as $key => $entity) {
            if ($entity instanceof CartCheckout) {
                $changes = $uow->getEntityChangeSet($entity);
                if (isset($changes['payedAt']) && $changes['payedAt'][1] != null) {
                    $items = $entity->getCart()->getItems();
                    /** @var CartItem $item */
                    $cartCheckoutPayedAt = $entity->getPayedAt();
                    $shopConfig = $this->shopConfigRepository->findOne();
                    $cartCheckoutPayedAt->add($shopConfig->getGiftCardValidTime());
                    foreach ($items as $item) {
                        if ($item->getVariant()->getType()->isGift()) {
                            for ($i = 0; $i < $item->getQuantity(); $i++) {
                                /** @var ProductGift $gift */
                                $gift = new ProductGift();
                                /** @var ShopConfig $shopConfig */
                                $gift->setValidUntilDate($cartCheckoutPayedAt);
                                $item->addGift($gift);
                                $metadata = $em->getClassMetadata(get_class($item));
                                $uow->computeChangeSet($metadata, $item);
                                $uow->persist($item);
                                $hasGift = true;
                            }
                        }
                    }
                    if ($hasGift) {
                        $contact = $em->getRepository(Contact::class)->findOne();
                        $this->mailer->sendGiftCard($contact, $entity->getCart());
                    }
                }
                if (isset($changes['payedAt']) && $changes['payedAt'][1] == null) {
                    $items = $entity->getCart()->getItems();
                    /** @var CartItem $item */
                    foreach ($items as $item) {
                        if ($item->getVariant()->getType()->isGift()) {
                            foreach ($item->getGifts() as $gift) {
                                $em->remove($gift);
                            }
                        }
                    }
                }
            }
        }
    }
}
