<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Season\Model;

use CpCreation\VitiCore\Behavior\HasTimestamp;
use CpCreation\VitiCore\Behavior\HasTranslatable;
use CpCreation\VitiCore\Behavior\HasVideo;
use CpCreation\VitiCore\Behavior\Impl\Timestamp;
use CpCreation\VitiCore\Behavior\Impl\Translatable;
use CpCreation\VitiCore\Behavior\Impl\Video;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Ramsey\Uuid\Uuid;
use Gedmo\Mapping\Annotation as Gedmo;
use Ramsey\Uuid\UuidInterface;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * Class Season
 *
 * @ORM\Entity(repositoryClass="CpCreation\VitiCore\Season\Repository\SeasonRepository")
 * @ORM\Table(name="season_season")
 * @ORM\HasLifecycleCallbacks()
 */
class Season implements HasTimestamp, HasVideo, HasTranslatable
{

    use Timestamp, Video, Translatable;

    /**
     * @ORM\Id()
     * @ORM\Column(type="uuid")
     * @ORM\GeneratedValue(strategy="NONE")
     *
     * @var UuidInterface
     */
    private $id;

    /**
     * @Gedmo\SortablePosition()
     * @ORM\Column(type="integer", nullable=true)
     *
     * @var int
     */
    private $position;

    /**
     * @ORM\OneToMany(targetEntity="CpCreation\VitiCore\Season\Model\SeasonImage", cascade={"persist","remove"}, mappedBy="season", orphanRemoval=true)
     * @ORM\OrderBy({"position" = "ASC"})
     * @Assert\Valid()
     *
     * @var Collection<SeasonImage>
     */
    private $images;

    public function __construct()
    {
        $this->id = Uuid::uuid4();
        $this->images = new ArrayCollection();
    }

    /**
     * @return UuidInterface
     */
    public function getId(): UuidInterface
    {
        return $this->id;
    }

    /**
     * @return int
     */
    public function getPosition(): int
    {
        return $this->position;
    }

    /**
     * @param int $position
     */
    public function setPosition(int $position)
    {
        $this->position = $position;
    }

    public function addImage(SeasonImage $image): self
    {

        if (!$this->images->contains($image)) {
            $this->images[] = $image;
            $image->setSeason($this);
        }

        return $this;
    }

    public function removeImage(SeasonImage $image): self
    {

        if ($this->images->contains($image)) {
            $this->images->removeElement($image);
            // set the owning side to null (unless already changed)
            if ($image->getSeason() === $this) {
                $image->setSeason(null);
            }
        }

        return $this;
    }

    /**
     * @return Collection<SeasonImage>
     */
    public function getImages(): Collection
    {
        return $this->images;
    }
}