<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\News\Model;

use CpCreation\VitiCore\Behavior\HasDisable;
use CpCreation\VitiCore\Behavior\HasFile;
use CpCreation\VitiCore\Behavior\HasTimestamp;
use CpCreation\VitiCore\Behavior\HasTranslatable;
use CpCreation\VitiCore\Behavior\HasVideo;
use CpCreation\VitiCore\Behavior\Impl\Disable;
use CpCreation\VitiCore\Behavior\Impl\File;
use CpCreation\VitiCore\Behavior\Impl\Timestamp;
use CpCreation\VitiCore\Behavior\Impl\Translatable;
use CpCreation\VitiCore\Behavior\Impl\Video;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Ramsey\Uuid\Uuid;
use Ramsey\Uuid\UuidInterface;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * Class News
 *
 * @ORM\Entity(repositoryClass="CpCreation\VitiCore\News\Repository\NewsRepository")
 * @ORM\Table(name="news_news")
 * @ORM\HasLifecycleCallbacks()
 */
class News implements HasTimestamp, HasDisable, HasVideo, HasFile, HasTranslatable
{
    use Timestamp, Disable, Video, File, Translatable;

    /**
     * @ORM\Id()
     * @ORM\Column(type="uuid")
     * @ORM\GeneratedValue(strategy="NONE")
     *
     * @var UuidInterface
     */
    private $id;

    /**
     * @ORM\Column(type="datetime", nullable=true)
     *
     * @var \DateTime
     */
    private $unPublishedAt;

    /**
     * @ORM\Column(type="datetime", nullable=true)
     *
     * @var \DateTime
     */
    private $publishedAt;

    /**
     * @ORM\Column(type="datetime", nullable=true)
     *
     * @var \DateTime | null
     */
    private $eventAt;

    /**
     * @ORM\Column(type="string", nullable=true)
     *
     * @var string
     */
    private $address;

    /**
     * @ORM\Column(type="string", nullable=true)
     *
     * @var string
     */
    private $postal;

    /**
     * @ORM\Column(type="string", nullable=true)
     *
     * @var string
     */
    private $city;

    /**
     * @ORM\Column(type="string", nullable=true)
     *
     * @var string
     */
    private $country;

    /**
     * @ORM\Column(type="string", nullable=true)
     *
     * @var string
     */
    private $author;

    /**
     * @ORM\OneToMany(targetEntity="CpCreation\VitiCore\News\Model\NewsImage", cascade={"persist","remove"}, mappedBy="news", orphanRemoval=true)
     * @ORM\OrderBy({"position" = "ASC"})
     * @Assert\Valid()
     *
     * @var Collection<NewsImage>
     */
    private $images;

    /**
     * @ORM\ManyToOne(targetEntity="Category", inversedBy="news")
     * @ORM\JoinColumn(name="news_category_id", referencedColumnName="id", nullable=true, onDelete="CASCADE")
     *
     * @var Category
     */
    private $category;

    public function __construct()
    {
        $this->id       = Uuid::uuid4();
        $this->disabled = 0;
        $this->images   = new ArrayCollection();
    }

    /**
     * @return UuidInterface
     */
    public function getId(): UuidInterface
    {
        return $this->id;
    }

    /**
     * @return \DateTime
     */
    public function getUnPublishedAt(): ?\DateTime
    {
        return $this->unPublishedAt;
    }

    /**
     * @param \DateTime $unPublishedAt
     */
    public function setUnPublishedAt(\DateTime $unPublishedAt = null)
    {
        $this->unPublishedAt = $unPublishedAt;
    }

    /**
     * @return \DateTime
     */
    public function getPublishedAt(): ?\DateTime
    {
        return $this->publishedAt;
    }

    /**
     * @param \DateTime $publishedAt
     */
    public function setPublishedAt(\DateTime $publishedAt = null): void
    {
        $this->publishedAt = $publishedAt;
    }

    /**
     * @return \DateTime|null
     */
    public function getEventAt(): ?\DateTime
    {
        return $this->eventAt;
    }

    /**
     * @param \DateTime|null $eventAt
     */
    public function setEventAt(?\DateTime $eventAt): void
    {
        $this->eventAt = $eventAt;
    }

    /**
     * @return string
     */
    public function getAddress(): ?string
    {
        return $this->address;
    }

    /**
     * @param string $address
     */
    public function setAddress(string $address = null): void
    {
        $this->address = $address;
    }

    /**
     * @return string
     */
    public function getPostal(): ?string
    {
        return $this->postal;
    }

    /**
     * @param string $postal
     */
    public function setPostal(string $postal = null)
    {
        $this->postal = $postal;
    }

    /**
     * @return string
     */
    public function getCity(): ?string
    {
        return $this->city;
    }

    /**
     * @param string $city
     */
    public function setCity(string $city = null)
    {
        $this->city = $city;
    }

    /**
     * @return string
     */
    public function getAuthor(): ?string
    {
        return $this->author;
    }

    /**
     * @param string $author
     */
    public function setAuthor(string $author = null)
    {
        $this->author = $author;
    }

    /**
     * @return string
     */
    public function getCountry(): ?string
    {
        return $this->country;
    }

    /**
     * @param string $country
     */
    public function setCountry(string $country = null)
    {
        $this->country = $country;
    }

    public function addImage(NewsImage $image): self
    {

        if (!$this->images->contains($image)) {
            $this->images[] = $image;
            $image->setNews($this);
        }

        return $this;
    }

    public function removeImage(NewsImage $image): self
    {

        if ($this->images->contains($image)) {
            $this->images->removeElement($image);
            // set the owning side to null (unless already changed)
            if ($image->getNews() === $this) {
                $image->setNews(null);
            }
        }

        return $this;
    }

    /**
     * @return Collection<Image>
     */
    public function getImages(): Collection
    {
        return $this->images;
    }

    /**
     * @return Category
     */
    public function getCategory(): ?Category
    {
        return $this->category;
    }

    /**
     * @param Category $category
     */
    public function setCategory(Category $category = null)
    {
        $this->category = $category;
    }

    /**
     * @ORM\PrePersist()
     * @ORM\PreUpdate()
     */
    public function setEventAtValue()
    {
        if (!$this->eventAt) {
            if (!$this->publishedAt) {
                $this->eventAt = new \DateTime();
            } else {
                $this->eventAt = $this->publishedAt;
            }
        }
    }
}
