<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Shop\Repository;

use CpCreation\VitiCore\Product\Model\ProductPriceGroup;
use CpCreation\VitiCore\Product\Model\ProductType;
use CpCreation\VitiCore\Repository\BaseRepository;
use CpCreation\VitiCore\Shop\Model\ShippingPrice;
use CpCreation\VitiCore\Shop\Model\ShippingZone;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\Persistence\ManagerRegistry;

class ShippingPriceRepository extends BaseRepository
{
    /**
     * ShippingPriceRepository constructor.
     * @param ManagerRegistry $registry
     * @throws \ReflectionException
     */
    public function __construct(ManagerRegistry $registry)
    {
        parent::__construct($registry, ShippingPrice::class);
    }

    /**
     * @return mixed
     */
    public function findTypesOfPrice()
    {
        $qb = $this
            ->createQueryBuilder('p')
            ->innerJoin('p.types', 't')
            ->innerJoin('t.translations', 'tr')
            ->groupBy('tr.name')
            ->getQuery()
            ->getResult();

        return $qb;
    }

    /**
     * @return mixed
     */
    public function findOrder()
    {
        $qb = $this
            ->createQueryBuilder('p')
            ->innerJoin('p.types', 't')
            ->innerJoin('t.translations', 'tr')
            ->addOrderBy('tr.name')
            ->addOrderBy('p.qtyAt')
            ->getQuery()
            ->getResult();

        return $qb;
    }

    /**
     * @return mixed
     * @throws \Doctrine\ORM\NonUniqueResultException
     */
    public function findFreesShipping()
    {
        $qb = $this
            ->createQueryBuilder('p')
            ->where('p.packagePrice = 0')
            ->getQuery()
            ->setMaxResults(1)
            ->getOneOrNullResult()
            ;

        return $qb;
    }

    /**
     * @param ProductPriceGroup $group
     * @param ProductType       $type
     * @return mixed
     * @throws \Doctrine\ORM\NonUniqueResultException
     */
    public function findFreesShippingByGroup(ProductPriceGroup $group, ProductType $type)
    {
        $qb = $this
            ->createQueryBuilder('p')
            ->where('p.packagePrice = 0')
            ->innerJoin('p.groups','g',Join::WITH,'g.id = :group')
            ->innerJoin('p.types', 't', Join::WITH,'t.id = :type')
            ->setParameter('group', $group->getId()->toString())
            ->setParameter('type', $type->getId()->toString())
            ->getQuery()
            ->setMaxResults(1)
            ->getOneOrNullResult()
        ;

        return $qb;
    }

    /**
     * @param int          $qty
     * @param string       $type
     * @param ShippingZone $zone
     * @return mixed
     * @throws \Doctrine\ORM\NonUniqueResultException
     */
    public function findByQuantityAndByTypeAndByZone(int $qty, string $type, ShippingZone $zone)
    {
        $qb = $this
            ->createQueryBuilder('p')
            ->innerJoin('p.types','t')
            ->innerJoin('t.translations', 'tr', Join::WITH,'tr.name = :type')
            ->innerJoin('p.zones','z',Join::WITH,'z.id = :zone')
            ->andwhere('p.qtyOf < :qty')
            ->andwhere('p.qtyAt >= :qty')
            ->setParameter('qty', $qty)
            ->setParameter('type', $type)
            ->setParameter('zone', $zone->getId()->toString())
            ->getQuery()
            ->setMaxResults(1)
            ->getOneOrNullResult();

        return $qb;
    }

    /**
     * @param int          $qty
     * @param string       $type
     * @param ShippingZone $zone
     * @param ProductPriceGroup $group
     * @return mixed
     * @throws \Doctrine\ORM\NonUniqueResultException
     */
    public function findByQuantityAndByTypeAndByZoneAndByGroup(int $qty, string $type, ShippingZone $zone, ProductPriceGroup $group)
    {

        $qb = $this
            ->createQueryBuilder('p')
            ->innerJoin('p.types','t')
            ->innerJoin('t.translations', 'tr', Join::WITH,'tr.name = :type')
            ->innerJoin('p.zones','z',Join::WITH,'z.id = :zone')
            ->innerJoin('p.groups','g',Join::WITH,'g.id = :group')
            ->andwhere('p.qtyOf < :qty')
            ->andwhere('p.qtyAt >= :qty')
            ->setParameter('qty', $qty)
            ->setParameter('type', $type)
            ->setParameter('group', $group->getId()->toString())
            ->setParameter('zone', $zone->getId()->toString())
            ->getQuery()
            ->setMaxResults(1)
            ->getOneOrNullResult();


        return $qb;
    }
}
