<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Shop\Model;

use CpCreation\VitiCore\Behavior\HasDisable;
use CpCreation\VitiCore\Behavior\HasTranslatable;
use CpCreation\VitiCore\Behavior\Impl\Disable;
use CpCreation\VitiCore\Behavior\Impl\Translatable;
use CpCreation\VitiCore\Product\Model\ProductPriceGroup;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Ramsey\Uuid\Uuid;
use Symfony\Component\Validator\Constraints as Assert;
use Gedmo\Mapping\Annotation as Gedmo;
use Symfony\Component\Validator\Context\ExecutionContextInterface;

/**
 * Class DeliveryMethod
 *
 * @ORM\Entity(repositoryClass="CpCreation\VitiCore\Shop\Repository\DeliveryMethodRepository")
 * @ORM\Table(name="shop_delivery_method")
 * @ORM\HasLifecycleCallbacks()
 */
class DeliveryMethod implements HasTranslatable, HasDisable
{
    use Translatable, Disable;

    /**
     * @ORM\Id()
     * @ORM\Column(type="uuid")
     * @ORM\GeneratedValue(strategy="NONE")
     *
     * @var Uuid
     */
    private $id;

    /**
     * @ORM\Column(type="datetime", nullable=true)
     *
     * @var \DateTime
     */
    private $unPublishedAt;

    /**
     * @ORM\Column(type="datetime", nullable=true)
     * @Assert\NotBlank()
     *
     * @var \DateTime
     */
    private $publishedAt;

    /**
     * @ORM\Column(type="integer", nullable=true)
     *
     * @var int
     */
    private $packagePrice;

    /**
     * @ORM\Column(type="integer", nullable=true)
     *
     * @var int
     */
    private $bottlePrice;

    /**
     * @ORM\Column(type="boolean")
     *
     * @var bool
     */
    private $viewCounties;

    /**
     * @ORM\Column(type="integer")
     * @Gedmo\SortablePosition()
     *
     * @var integer
     */
    private $position;

    /**
     * @ORM\ManyToOne(targetEntity="DeliveryMethodCategory", inversedBy="methods")
     * @ORM\JoinColumn(name="delivery_category_id", referencedColumnName="id", nullable=true)
     * @Gedmo\SortableGroup()
     *
     * @Assert\NotBlank()
     * @var DeliveryMethodCategory
     */
    private $category;

    /**
     * @ORM\ManyToMany(targetEntity="ShippingCounty")
     * @ORM\JoinTable(name="shop_delivery_method_counties",
     *      joinColumns={@ORM\JoinColumn(name="deliverymethod_id", referencedColumnName="id")},
     *      inverseJoinColumns={@ORM\JoinColumn(name="countie_id", referencedColumnName="id", unique=true)})
     * @ORM\OrderBy({"code" = "ASC"})
     * @Assert\Valid()
     *
     * @var Collection<ShippingCounty>
     */
    private $counties;

    /**
     * @ORM\ManyToMany(targetEntity="CpCreation\VitiCore\Product\Model\ProductPriceGroup", fetch="EAGER")
     * @ORM\JoinTable(name="shop_delivery_method_group_price")
     * @Assert\Count( min = 1 , minMessage="Vous devez ajouter au moins un groupe")
     * @var Collection<ProductPriceGroup>
     */
    private $groups;

    /**
     * DeliveryMethod constructor.
     * @throws \Exception
     */
    public function __construct()
    {
        $this->id = Uuid::uuid4();
        $this->disabled = false;
        $this->counties = new ArrayCollection();
        $this->groups = new ArrayCollection();
        $this->viewCounties = false;
    }

    /**
     * @return Uuid
     */
    public function getId(): Uuid
    {
        return $this->id;
    }

    /**
     * @return \DateTime
     */
    public function getUnPublishedAt(): ?\DateTime
    {
        return $this->unPublishedAt;
    }

    /**
     * @param \DateTime $unPublishedAt
     */
    public function setUnPublishedAt(\DateTime $unPublishedAt = null)
    {
        $this->unPublishedAt = $unPublishedAt;
    }

    /**
     * @return \DateTime
     */
    public function getPublishedAt(): ?\DateTime
    {
        return $this->publishedAt;
    }

    /**
     * @param \DateTime $publishedAt
     */
    public function setPublishedAt(\DateTime $publishedAt = null)
    {
        $this->publishedAt = $publishedAt;
    }

    /**
     * @return int
     */
    public function getPackagePrice(): ?int
    {
        return $this->packagePrice;
    }

    /**
     * @param int $packagePrice
     */
    public function setPackagePrice(int $packagePrice): void
    {
        $this->packagePrice = $packagePrice;
    }

    /**
     * @return int
     */
    public function getBottlePrice(): ?int
    {
        return $this->bottlePrice;
    }

    /**
     * @param int $bottlePrice
     */
    public function setBottlePrice(int $bottlePrice): void
    {
        $this->bottlePrice = $bottlePrice;
    }

    /**
     * @return bool
     */
    public function isViewCounties(): bool
    {
        return $this->viewCounties;
    }

    /**
     * @param bool $viewCounties
     */
    public function setViewCounties(bool $viewCounties)
    {
        $this->viewCounties = $viewCounties;
    }

    /**
     * @return int
     */
    public function getPosition(): ?int
    {
        return $this->position;
    }

    /**
     * @param int $position
     */
    public function setPosition(int $position)
    {
        $this->position = $position;
    }

    /**
     * @return DeliveryMethodCategory
     */
    public function getCategory(): ?DeliveryMethodCategory
    {
        return $this->category;
    }

    /**
     * @param DeliveryMethodCategory $category
     */
    public function setCategory(DeliveryMethodCategory $category = null)
    {
        $this->category = $category;
    }

    /**
     * @param ShippingCounty $county
     * @return $this
     */
    public function addCounty(ShippingCounty $county)
    {
        $this->counties[] = $county;

        return $this;
    }

    /**
     * @param ShippingCounty $county
     */
    public function removeCounty(ShippingCounty $county)
    {
        $this->counties->removeElement($county);
    }

    /**
     * @return Collection<ShippingCounty>
     */
    public function getCounties(): Collection
    {
        return $this->counties;
    }

    /**
     * @param ProductPriceGroup $group
     * @return $this
     */
    public function addGroup(ProductPriceGroup $group)
    {
        $this->groups[] = $group;

        return $this;
    }

    /**
     * @param ProductPriceGroup $group
     */
    public function removeGroup(ProductPriceGroup $group)
    {
        $this->groups->removeElement($group);
    }

    /**
     * @return Collection<ProductPriceGroup>
     */
    public function getGroups(): Collection
    {
        return $this->groups;
    }

    /**
     * @param \Symfony\Component\Validator\Context\ExecutionContextInterface $context
     * @Assert\Callback()
     */
    public function validate(ExecutionContextInterface $context)
    {
        if (is_null($this->getPackagePrice()) && is_null($this->getBottlePrice())) {
            $message = 'Veuillez ajouter un prix';
            $context->buildViolation($message)
                    ->atPath('bottlePrice')
                    ->addViolation();
        }
    }
}