<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Product\Repository;

use CpCreation\VitiCore\Product\Model\Product;
use CpCreation\VitiCore\Product\Model\ProductCategory;
use CpCreation\VitiCore\Product\Model\ProductPriceGroup;
use CpCreation\VitiCore\Repository\BaseRepository;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\Persistence\ManagerRegistry;

class ProductRepository extends BaseRepository
{
    /**
     * ProductRepository constructor.
     * @param ManagerRegistry $registry
     * @throws \ReflectionException
     */
    public function __construct(ManagerRegistry $registry)
    {
        parent::__construct($registry, Product::class);
    }

    public function findAllNotDeleted()
    {
        $qb = $this->createQueryBuilder('product')
                   ->addSelect('va')
                   ->innerJoin('product.variants', 'va', Join::WITH, 'va.deletedAt IS NULL')
                   ->orderBy('product.position', 'ASC');


        return $qb->getQuery()->getResult();
    }

    public function findAllPublish(ProductPriceGroup $group = null, ProductCategory $category = null)
    {
        $qb = $this->createQueryBuilder('product')
                   ->addSelect('tr, im, im2, ca, va, ty, tytr, pr')
                   ->innerJoin('product.translations', 'tr', Join::WITH, 'tr.translatable = product.id')
                   ->leftJoin('product.category', 'ca', Join::WITH, 'ca.id = product.category')
                   ->innerJoin('product.image', 'im', Join::WITH, 'im.id = product.image')
                   ->leftJoin('product.secondImage', 'im2', Join::WITH, 'im2.id = product.secondImage')
                   ->innerJoin('product.variants', 'va', Join::WITH, 'va.unPublished = FALSE')
                   ->innerJoin('va.prices', 'pr')
                   ->innerJoin('va.type', 'ty')
                   ->innerJoin('ty.translations', 'tytr')
                   ->andWhere('product.disabled = FALSE')
                   ->orderBy('product.position', 'ASC');


        if ($category) {
            $qb
                ->andWhere('product.category = :category')
                ->setParameter('category', $category);
        }

        if ($group) {
            $qb
                ->innerJoin('product.groups', 'gr', Join::WITH, 'gr = :group')
                ->setParameter('group', $group);
        }

        return $qb->getQuery()->getResult();
    }

    public function findOnlyCuvee($category = null)
    {
        if ($category) {
            $qb = $this->createQueryBuilder('product')
                       ->addSelect('tr,im')
                       ->innerJoin('product.translations', 'tr', Join::WITH, 'tr.translatable = product.id')
                       ->innerJoin('product.image', 'im', Join::WITH, 'im.id = product.image')
                       ->andWhere('product.category = :category')
                       ->andWhere('product.disabled = FALSE')
                       ->andWhere('product.cuvee = TRUE')
                       ->orderBy('product.position', 'ASC')
                       ->setParameter('category', $category)
                       ->getQuery();

            return $qb->execute();
        }

        $qb = $this->createQueryBuilder('product')
                   ->addSelect('tr, im')
                   ->innerJoin('product.translations', 'tr')
                   ->innerJoin('product.image', 'im')
                   ->andWhere('product.disabled = FALSE')
                   ->andWhere('product.cuvee = TRUE')
                   ->orderBy('product.position', 'ASC')
                   ->getQuery();

        return $qb->execute();

    }

    public function findOnlyShop(ProductPriceGroup $group = null, $category = null)
    {
        $qb = $this->createQueryBuilder('product')
                   ->addSelect('tr,im')
                   ->innerJoin('product.translations', 'tr', Join::WITH, 'tr.translatable = product.id')
                   ->innerJoin('product.image', 'im', Join::WITH, 'im.id = product.image')
                   ->innerJoin('product.variants', 'va', Join::WITH, 'va.unPublished = FALSE OR va.deletedAt IS NULL')
                   ->andWhere('product.disabled = FALSE')
                   ->andWhere('product.shop = TRUE')
                   ->orderBy('product.position', 'ASC');


        if ($category) {
            $qb
                ->andWhere('product.category = :category')
                ->setParameter('category', $category);
        }

        if ($group) {
            $qb
                ->innerJoin('product.groups', 'gr', Join::WITH, 'gr = :group')
                ->setParameter('group', $group);
        }

        return $qb->getQuery()->getResult();

    }
}
