<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Product\Model;

use CpCreation\VitiCore\Behavior\HasTimestamp;
use CpCreation\VitiCore\Behavior\Impl\Timestamp;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Ramsey\Uuid\Uuid;
use Symfony\Component\Validator\Constraints as Assert;
use Symfony\Component\Validator\Context\ExecutionContextInterface;

/**
 * Class ProductDegressive
 *
 * @ORM\Entity(repositoryClass="CpCreation\VitiCore\Product\Repository\ProductDegressiveRepository")
 * @ORM\Table(name="product_degressive")
 * @ORM\HasLifecycleCallbacks()
 */
class ProductDegressive implements HasTimestamp
{
    use Timestamp;

    /**
     * @ORM\Id()
     * @ORM\Column(type="uuid")
     * @ORM\GeneratedValue(strategy="NONE")
     *
     * @var Uuid
     */
    private $id;

    /**
     * @ORM\Column(type="integer")
     * @Assert\NotBlank()
     *
     * @var integer
     */
    private $qtyOf;

    /**
     * @ORM\Column(type="integer")
     * @Assert\NotBlank()
     *
     * @var integer
     */
    private $qtyAt;

    /**
     * @ORM\Column(type="integer", nullable=true)
     *
     * @var int
     */
    private $packagePrice;

    /**
     * @ORM\Column(type="integer", nullable=true)
     *
     * @var int
     */
    private $bottlePrice;

    /**
     * @ORM\Column(type="decimal", precision=5, scale=3, nullable=true)
     *
     * @var float | null
     */
    private $packagePercent;

    /**
     * @ORM\Column(type="integer", nullable=true)
     *
     * @var int|null
     */
    private $qtyOfBottleOffered;

    /**
     * @ORM\Column(type="boolean")
     *
     * @var boolean
     */
    private $equivalency;

    /**
     * @ORM\Column(type="boolean")
     *
     * @var boolean
     */
    private $allQuantity;

    /**
     * @ORM\ManyToMany(targetEntity="CpCreation\VitiCore\Product\Model\ProductType", fetch="EAGER")
     * @Assert\Count( min = 1 , minMessage="Vous devez ajouter au moins un type")
     * @var Collection<ProductType>
     */
    private $types;

    /**
     * @ORM\ManyToMany(targetEntity="CpCreation\VitiCore\Product\Model\ProductPriceGroup", fetch="EAGER")
     * @Assert\Count( min = 1 , minMessage="Vous devez ajouter au moins un groupe")
     * @var Collection<ProductPriceGroup>
     */
    private $groups;

    public function __construct()
    {
        $this->id          = Uuid::uuid4();
        $this->types       = new ArrayCollection();
        $this->groups      = new ArrayCollection();
        $this->allQuantity = false;
        $this->equivalency = false;
    }

    /**
     * @return Uuid
     */
    public function getId(): Uuid
    {
        return $this->id;
    }

    /**
     * @return int
     */
    public function getQtyOf(): ?int
    {
        return $this->qtyOf;
    }

    /**
     * @param int $qtyOf
     */
    public function setQtyOf(int $qtyOf): void
    {
        $this->qtyOf = $qtyOf;
    }

    /**
     * @return int
     */
    public function getQtyAt(): ?int
    {
        return $this->qtyAt;
    }

    /**
     * @param int $qtyAt
     */
    public function setQtyAt(int $qtyAt): void
    {
        $this->qtyAt = $qtyAt;
    }

    /**
     * @return int
     */
    public function getPackagePrice(): ?int
    {
        return $this->packagePrice;
    }

    /**
     * @param int $packagePrice
     */
    public function setPackagePrice(int $packagePrice = null): void
    {
        $this->packagePrice = $packagePrice;
    }

    /**
     * @return int
     */
    public function getBottlePrice(): ?int
    {
        return $this->bottlePrice;
    }

    /**
     * @param int $bottlePrice
     */
    public function setBottlePrice(int $bottlePrice = null): void
    {
        $this->bottlePrice = $bottlePrice;
    }

    /**
     * @return float|null
     */
    public function getPackagePercent(): ?float
    {
        return $this->packagePercent;
    }

    /**
     * @param float|null $packagePercent
     */
    public function setPackagePercent(?float $packagePercent): void
    {
        $this->packagePercent = $packagePercent;
    }

    /**
     * @return bool
     */
    public function isEquivalency(): bool
    {
        return $this->equivalency;
    }

    /**
     * @param bool $equivalency
     */
    public function setEquivalency(bool $equivalency): void
    {
        $this->equivalency = $equivalency;
    }

    /**
     * @return bool
     */
    public function isAllQuantity(): bool
    {
        return $this->allQuantity;
    }

    /**
     * @param bool $allQuantity
     */
    public function setAllQuantity(bool $allQuantity): void
    {
        $this->allQuantity = $allQuantity;
    }

    /**
     * @return int|null
     */
    public function getQtyOfBottleOffered(): ?int
    {
        return $this->qtyOfBottleOffered;
    }

    /**
     * @param int|null $qtyOfBottleOffered
     */
    public function setQtyOfBottleOffered(?int $qtyOfBottleOffered): void
    {
        $this->qtyOfBottleOffered = $qtyOfBottleOffered;
    }

    /**
     * @param ProductType $productType
     * @return $this
     */
    public function addType(ProductType $productType)
    {
        $this->types[] = $productType;

        return $this;
    }

    /**
     * @param ProductType $productType
     */
    public function removeType(ProductType $productType)
    {
        $this->types->removeElement($productType);
    }

    /**
     * @return Collection<ProductType>
     */
    public function getTypes(): Collection
    {
        return $this->types;
    }

    /**
     * @param ProductPriceGroup $group
     * @return $this
     */
    public function addGroup(ProductPriceGroup $group)
    {
        $this->groups[] = $group;

        return $this;
    }

    /**
     * @param ProductPriceGroup $group
     */
    public function removeGroup(ProductPriceGroup $group)
    {
        $this->groups->removeElement($group);
    }

    /**
     * @return Collection<ProductPriceGroup>
     */
    public function getGroups(): Collection
    {
        return $this->groups;
    }

    /**
     * @param \Symfony\Component\Validator\Context\ExecutionContextInterface $context
     * @Assert\Callback()
     */
    public function validate(ExecutionContextInterface $context)
    {
        if (is_null($this->getPackagePrice()) && is_null($this->getBottlePrice()) && is_null($this->getPackagePercent()) && is_null($this->getQtyOfBottleOffered())) {
            $message = 'Veuillez ajouter un prix';
            $context->buildViolation($message)
                    ->atPath('bottlePrice')
                    ->addViolation();
        }
    }
}
