<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Product\Model;

use CpCreation\VitiCore\Award\Model\Award;
use CpCreation\VitiCore\Behavior\HasDisable;
use CpCreation\VitiCore\Behavior\HasFile;
use CpCreation\VitiCore\Behavior\HasImage;
use CpCreation\VitiCore\Behavior\HasTimestamp;
use CpCreation\VitiCore\Behavior\HasTranslatable;
use CpCreation\VitiCore\Behavior\Impl\Disable;
use CpCreation\VitiCore\Behavior\Impl\File;
use CpCreation\VitiCore\Behavior\Impl\Image;
use CpCreation\VitiCore\Behavior\Impl\Timestamp;
use CpCreation\VitiCore\Behavior\Impl\Translatable;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Sortable\Sortable;
use Ramsey\Uuid\Uuid;
use Symfony\Component\Validator\Constraints as Assert;
use Gedmo\Mapping\Annotation as Gedmo;
use CpCreation\VitiCore\Media\Model\Image as ModelImage;
use Symfony\Component\Validator\Context\ExecutionContextInterface;

/**
 * Class Product
 *
 * @ORM\Entity(repositoryClass="CpCreation\VitiCore\Product\Repository\ProductRepository")
 * @ORM\Table(name="product_product")
 * @ORM\HasLifecycleCallbacks()
 */
class Product implements HasTimestamp, HasImage, HasFile, Sortable, HasDisable, HasTranslatable
{
    use Timestamp, Image, File, Disable, Translatable;

    /**
     * @ORM\Id()
     * @ORM\Column(type="uuid")
     * @ORM\GeneratedValue(strategy="NONE")
     *
     * @var Uuid
     */
    private $id;

    /**
     * @ORM\Column(type="boolean")
     *
     * @var bool
     */
    private $cuvee;

    /**
     * @ORM\Column(type="boolean")
     *
     * @var bool
     */
    private $shop;

    /**
     * @ORM\Column(type="integer")
     * @Gedmo\SortablePosition()
     *
     * @var integer
     */
    private $position;

    /**
     * @ORM\Column(type="boolean")
     *
     * @var boolean
     */
    private $new;

    /**
     * @ORM\Column(type="boolean")
     *
     * @var boolean
     */
    private $soldOut;

    /**
     * @ORM\Column(type="text", nullable=true, length=8)
     *
     * @var string | null
     */
    private $ribbonCustomColor;

    /**
     * @ORM\Column(type="boolean")
     *
     * @var boolean
     */
    private $enableColor;

    /**
     * @ORM\Column(type="text", nullable=true, length=8)
     *
     * @var string | null
     */
    private $color;

    /**
     * @ORM\ManyToOne(targetEntity="ProductCategory", inversedBy="products")
     * @ORM\JoinColumn(name="product_category_id", referencedColumnName="id", nullable=true)
     *
     * @var ProductCategory
     */
    private $category;

    /**
     * @ORM\OneToMany(targetEntity="ProductVariant", mappedBy="product", cascade={"persist","remove"}, orphanRemoval=true)
     * @ORM\OrderBy({"position" = "ASC"})
     *
     * @Assert\Count( min = 1 , minMessage="Vous devez ajouter au moins un variant de produit")
     * @Assert\Valid()
     *
     * @var Collection<ProductVariant>
     */
    private $variants;

    /**
     * @ORM\ManyToMany(targetEntity="CpCreation\VitiCore\Award\Model\Award")
     * @Assert\Valid()
     *
     * @var Collection<Award>
     */
    private $awards;

    /**
     * @ORM\OneToOne(targetEntity="CpCreation\VitiCore\Media\Model\Image", cascade={"persist", "remove"})
     * @var ModelImage
     */
    private $secondImage;

    /**
     * @ORM\ManyToMany(targetEntity="CpCreation\VitiCore\Product\Model\ProductPriceGroup", fetch="EAGER")
     * @var Collection<ProductPriceGroup>
     */
    private $groups;

    public function __construct()
    {
        $this->id          = Uuid::uuid4();
        $this->variants    = new ArrayCollection();
        $this->awards      = new ArrayCollection();
        $this->groups      = new ArrayCollection();
        $this->disabled    = false;
        $this->cuvee       = true;
        $this->shop        = true;
        $this->new         = false;
        $this->soldOut     = false;
        $this->enableColor = false;

    }

    /**
     * @return Uuid
     */
    public function getId(): Uuid
    {
        return $this->id;
    }

    /**
     * @param Uuid $id
     */
    public function setId(Uuid $id)
    {
        $this->id = $id;
    }

    /**
     * @return bool
     */
    public function isCuvee(): bool
    {
        return $this->cuvee;
    }

    /**
     * @param bool $cuvee
     */
    public function setCuvee(bool $cuvee = true)
    {
        $this->cuvee = $cuvee;
    }

    /**
     * @return bool
     */
    public function isShop(): bool
    {
        return $this->shop;
    }

    /**
     * @param bool $shop
     */
    public function setShop(bool $shop)
    {
        $this->shop = $shop;
    }

    /**
     * @return int
     */
    public function getPosition(): ?int
    {
        return $this->position;
    }

    /**
     * @return bool
     */
    public function isNew(): bool
    {
        return $this->new;
    }

    /**
     * @param bool $new
     */
    public function setNew(bool $new)
    {
        $this->new = $new;
    }

    /**
     * @return bool
     */
    public function isSoldOut(): bool
    {
        return $this->soldOut;
    }

    /**
     * @param bool $soldOut
     */
    public function setSoldOut(bool $soldOut): void
    {
        $this->soldOut = $soldOut;
    }

    /**
     * @return null|string
     */
    public function getRibbonCustomColor(): ?string
    {
        return $this->ribbonCustomColor;
    }

    /**
     * @param null|string $ribbonCustomColor
     */
    public function setRibbonCustomColor(?string $ribbonCustomColor): void
    {
        $this->ribbonCustomColor = $ribbonCustomColor;
    }

    /**
     * @return bool
     */
    public function isEnableColor(): bool
    {
        return $this->enableColor;
    }

    /**
     * @param bool $enableColor
     */
    public function setEnableColor(bool $enableColor): void
    {
        $this->enableColor = $enableColor;
    }

    /**
     * @return null|string
     */
    public function getColor(): ?string
    {
        return $this->color;
    }

    /**
     * @param null|string $color
     */
    public function setColor(?string $color): void
    {
        $this->color = $color;
    }

    /**
     * @param int $position | null
     */
    public function setPosition(?int $position)
    {
        $this->position = $position;
    }

    /**
     * @return ProductCategory
     */
    public function getCategory(): ?ProductCategory
    {
        return $this->category;
    }

    /**
     * @param ProductCategory $category
     */
    public function setCategory(ProductCategory $category = null)
    {
        $this->category = $category;
    }

    /**
     * @param ProductVariant $variant
     * @return $this
     */
    public function addVariant(ProductVariant $variant)
    {
        $variant->setProduct($this);
        $this->variants[] = $variant;

        return $this;
    }

    /**
     * @param ProductVariant $variant
     */
    public function removeVariant(ProductVariant $variant)
    {
        $variant->markDeleted();
        $this->variants[] = $variant;

        return $this;
    }

    /**
     * @return Collection<ProductVariant>
     */
    public function getVariants(): Collection
    {
        return $this->variants;
    }

    public function addAward(Award $award)
    {
        $this->awards[] = $award;
    }

    public function removeAward(Award $award)
    {
        $this->awards->removeElement($award);
    }

    /**
     * @return Collection<Award>
     */
    public function getAwards(): Collection
    {
        return $this->awards;
    }

    /**
     * @return ModelImage
     */
    public function getSecondImage(): ?ModelImage
    {
        return $this->secondImage;
    }

    /**
     * @param ProductPriceGroup $group
     * @return $this
     */
    public function addGroup(ProductPriceGroup $group)
    {
        $this->groups[] = $group;

        return $this;
    }

    /**
     * @param ProductPriceGroup $group
     */
    public function removeGroup(ProductPriceGroup $group)
    {
        $this->groups->removeElement($group);
    }

    /**
     * @return Collection<ProductPriceGroup>
     */
    public function getGroups(): Collection
    {
        return $this->groups;
    }

    /**
     * @param ModelImage $secondImage
     */
    public function setSecondImage(ModelImage $secondImage = null)
    {
        if ($secondImage->getPath() || $secondImage->getFile()) {
            $this->secondImage = $secondImage;
        } else {
            $this->secondImage = null;
        }
    }

    /**
     * @param \Symfony\Component\Validator\Context\ExecutionContextInterface $context
     * @Assert\Callback()
     */
    public function validate(ExecutionContextInterface $context)
    {
        if (!$this->getImage()) {
            $message = 'Veuillez ajouter une image';
            $context->buildViolation($message)
                    ->atPath("image.file")
                    ->addViolation();
        }
    }
}
