<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\History\Model;

use CpCreation\VitiCore\Behavior\HasDisable;
use CpCreation\VitiCore\Behavior\HasFile;
use CpCreation\VitiCore\Behavior\HasTimestamp;
use CpCreation\VitiCore\Behavior\HasTranslatable;
use CpCreation\VitiCore\Behavior\HasVideo;
use CpCreation\VitiCore\Behavior\Impl\Disable;
use CpCreation\VitiCore\Behavior\Impl\File;
use CpCreation\VitiCore\Behavior\Impl\Timestamp;
use CpCreation\VitiCore\Behavior\Impl\Translatable;
use CpCreation\VitiCore\Behavior\Impl\Video;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Ramsey\Uuid\Uuid;
use Gedmo\Mapping\Annotation as Gedmo;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * Class History
 *
 * @ORM\Entity(repositoryClass="CpCreation\VitiCore\History\Repository\HistoryRepository")
 * @ORM\Table(name="history_history")
 * @ORM\HasLifecycleCallbacks()
 */
class History implements HasTimestamp, HasDisable, HasVideo, HasFile, HasTranslatable
{
    use Timestamp, Disable, Video, File, Translatable;

    /**
     * @ORM\Id()
     * @ORM\Column(type="uuid")
     * @ORM\GeneratedValue(strategy="NONE")
     *
     * @var Uuid
     */
    private $id;

    /**
     * @ORM\Column(type="integer")
     * @Gedmo\SortablePosition()
     *
     * @var integer
     */
    private $position;

    /**
     * @ORM\OneToMany(targetEntity="CpCreation\VitiCore\History\Model\HistoryImage", cascade={"persist","remove"}, mappedBy="history", orphanRemoval=true)
     * @ORM\OrderBy({"position" = "ASC"})
     * @Assert\Valid()
     *
     * @var Collection<HistoryImage>
     */
    private $images;

    public function __construct()
    {
        $this->id       = Uuid::uuid4();
        $this->disabled = 0;
        $this->images   = new ArrayCollection();
    }

    /**
     * @return Uuid
     */
    public function getId(): Uuid
    {
        return $this->id;
    }

    /**
     * @return int
     */
    public function getPosition(): int
    {
        return $this->position;
    }

    /**
     * @param int $position
     */
    public function setPosition(int $position): void
    {
        $this->position = $position;
    }

    public function addImage(HistoryImage $image): self
    {

        if (!$this->images->contains($image)) {
            $this->images[] = $image;
            $image->setHistory($this);
        }

        return $this;
    }

    public function removeImage(HistoryImage $image): self
    {

        if ($this->images->contains($image)) {
            $this->images->removeElement($image);
            // set the owning side to null (unless already changed)
            if ($image->getHistory() === $this) {
                $image->setHistory(null);
            }
        }

        return $this;
    }

    /**
     * @return Collection<Image>
     */
    public function getImages(): Collection
    {
        return $this->images;
    }

    /**
     * @ORM\PrePersist()
     * @ORM\PreUpdate()
     */
    public function ImagesUpdate()
    {
        /** @var HistoryImage $image */
        foreach ($this->getImages() as $image) {
            $image->setHistory($this);
        }
    }
}
