<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Gite\Model;

use CpCreation\VitiCore\Behavior\HasFile;
use CpCreation\VitiCore\Behavior\HasTimestamp;
use CpCreation\VitiCore\Behavior\HasTranslatable;
use CpCreation\VitiCore\Behavior\HasVideo;
use CpCreation\VitiCore\Behavior\Impl\File;
use CpCreation\VitiCore\Behavior\Impl\Timestamp;
use CpCreation\VitiCore\Behavior\Impl\Translatable;
use CpCreation\VitiCore\Behavior\Impl\Video;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Ramsey\Uuid\Uuid;
use Gedmo\Mapping\Annotation as Gedmo;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * Class Room
 *
 * @ORM\Entity(repositoryClass="CpCreation\VitiCore\Gite\Repository\RoomRepository")
 * @ORM\Table(name="gite_room")
 * @ORM\HasLifecycleCallbacks()
 */
class Room implements HasTimestamp, HasVideo, HasFile, HasTranslatable
{
    use Timestamp, Video, File, Translatable;

    /**
     * @ORM\Id()
     * @ORM\Column(type="uuid")
     * @ORM\GeneratedValue(strategy="NONE")
     *
     * @var Uuid
     */
    private $id;

    /**
     * @ORM\Column(type="integer")
     * @Gedmo\SortablePosition()
     *
     * @var integer
     */
    private $position;

    /**
     * @ORM\ManyToOne(targetEntity="Gite", inversedBy="rooms")
     * @ORM\JoinColumn(name="gite_id", referencedColumnName="id", onDelete="cascade")
     *
     * @var Gite
     */
    private $gite;

    /**
     * @ORM\OneToMany(targetEntity="CpCreation\VitiCore\Gite\Model\RoomImage", cascade={"persist","remove"}, mappedBy="room", orphanRemoval=true)
     * @ORM\OrderBy({"position" = "ASC"})
     * @Assert\Count( min = 1 , minMessage="Vous devez ajouter au moins une image")
     * @Assert\Valid()
     *
     * @var Collection<RoomImage>
     */
    private $images;

    public function __construct()
    {
        $this->id       = Uuid::uuid4();
        $this->images   = new ArrayCollection();
    }

    /**
     * @return Uuid
     */
    public function getId(): Uuid
    {
        return $this->id;
    }

    /**
     * @return int
     */
    public function getPosition(): int
    {
        return $this->position;
    }

    /**
     * @param int $position
     */
    public function setPosition(int $position): void
    {
        $this->position = $position;
    }

    /**
     * @return Gite
     */
    public function getGite(): ?Gite
    {
        return $this->gite;
    }

    /**
     * @param Gite $gite
     */
    public function setGite(Gite $gite = null): void
    {
        $this->gite = $gite;
    }

    public function addImage(RoomImage $image): self
    {

        if (!$this->images->contains($image)) {
            $this->images[] = $image;
            $image->setRoom($this);
        }

        return $this;
    }

    public function removeImage(RoomImage $image): self
    {

        if ($this->images->contains($image)) {
            $this->images->removeElement($image);
            // set the owning side to null (unless already changed)
            if ($image->getRoom() === $this) {
                $image->setRoom(null);
            }
        }

        return $this;
    }

    /**
     * @return Collection<Image>
     */
    public function getImages(): Collection
    {
        return $this->images;
    }
}
