<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Cart\Model;

use CpCreation\VitiCore\Behavior\HasDisable;
use CpCreation\VitiCore\Behavior\HasTimestamp;
use CpCreation\VitiCore\Behavior\Impl\Disable;
use CpCreation\VitiCore\Behavior\Impl\Timestamp;
use Doctrine\ORM\Mapping as ORM;
use Ramsey\Uuid\Uuid;
use Symfony\Component\Validator\Constraints as Assert;
use Symfony\Component\Validator\Context\ExecutionContextInterface;

/**
 * Class Cart
 *
 * @ORM\Entity(repositoryClass="CpCreation\VitiCore\Cart\Repository\PromoRepository")
 * @ORM\Table(name="cart_promo")
 * @ORM\HasLifecycleCallbacks()
 */
class Promo implements HasTimestamp, HasDisable
{
    use Timestamp, Disable;

    /**
     * @ORM\Id()
     * @ORM\Column(type="uuid")
     * @ORM\GeneratedValue(strategy="NONE")
     *
     * @var Uuid
     */
    private $id;

    /**
     * @ORM\Column(type="string")
     * @Assert\NotBlank()
     *
     * @var string
     */
    private $title;

    /**
     * @ORM\Column(type="decimal", precision=3, scale=2, nullable=true)
     *
     * @var float | null
     */
    private $percent;

    /**
     * @ORM\Column(type="integer", nullable=true)
     *
     * @var integer | null
     */
    private $amount;

    /**
     * @ORM\Column(type="datetime")
     * @Assert\NotBlank()
     *
     * @var \DateTime
     */
    private $beginAt;

    /**
     * @ORM\Column(type="datetime")
     * @Assert\NotBlank()
     *
     * @var \DateTime
     */
    private $endAt;

    /**
     * @var integer
     *
     * @ORM\Column(type="integer", options={"default" : 0})
     */
    private $count = 0;

    /**
     * @var integer | null
     *
     * @ORM\Column(type="integer", nullable=true)
     */
    private $maxCount;

    /**
     * @var integer | null
     *
     * @ORM\Column(type="integer", nullable=true)
     */
    private $minEquivalency;

    public function __construct()
    {
        $this->id       = Uuid::uuid4();
        $this->disabled = false;
    }

    /**
     * @return Uuid
     */
    public function getId(): Uuid
    {
        return $this->id;
    }

    /**
     * @return string
     */
    public function getTitle(): ?string
    {
        return $this->title;
    }

    /**
     * @param string $title
     */
    public function setTitle(string $title): void
    {
        $this->title = $title;
    }

    /**
     * @return int|null
     */
    public function getAmount(): ?int
    {
        return $this->amount;
    }

    /**
     * @param int|null $amount
     */
    public function setAmount(?int $amount): void
    {
        $this->amount = $amount;
    }

    /**
     * @return float
     */
    public function getPercent(): ?float
    {
        return $this->percent;
    }

    /**
     * @param float $percent
     */
    public function setPercent(?float $percent): void
    {
        $this->percent = $percent;
    }

    /**
     * @return int
     */
    public function getCount(): int
    {
        return $this->count;
    }

    /**
     * @param int $count
     */
    public function setCount(int $count): void
    {
        $this->count = $count;
    }

    /**
     * @return int
     */
    public function getMaxCount(): ?int
    {
        return $this->maxCount;
    }

    /**
     * @param int $maxCount
     */
    public function setMaxCount(?int $maxCount): void
    {
        $this->maxCount = $maxCount;
    }

    /**
     * @return \DateTime
     */
    public function getBeginAt(): ?\DateTime
    {
        return $this->beginAt;
    }

    /**
     * @param \DateTime $beginAt
     */
    public function setBeginAt(\DateTime $beginAt = null): void
    {
        $this->beginAt = $beginAt;
    }

    /**
     * @return \DateTime
     */
    public function getEndAt(): ?\DateTime
    {
        return $this->endAt;
    }

    /**
     * @param \DateTime $endAt
     */
    public function setEndAt(\DateTime $endAt = null): void
    {
        $this->endAt = $endAt;
    }

    /**
     * @return int|null
     */
    public function getMinEquivalency(): ?int
    {
        return $this->minEquivalency;
    }

    /**
     * @param int|null $minEquivalency
     */
    public function setMinEquivalency(?int $minEquivalency): void
    {
        $this->minEquivalency = $minEquivalency;
    }

    public function incrementCount()
    {
        $this->count = $this->count++;
    }

    /**
     * @param \Symfony\Component\Validator\Context\ExecutionContextInterface $context
     * @Assert\Callback()
     */
    public function validate(ExecutionContextInterface $context)
    {
        if (is_null($this->getPercent()) && is_null($this->getAmount())) {
            $message = 'Veuillez entrer un pourcentage de remise ou un montant de remise';
            $context->buildViolation($message)
                    ->atPath('percent')
                    ->addViolation();
        }

        if ($this->getPercent() && $this->getAmount()) {
            $message = 'Veuillez entrer soit un pourcentage de remise ou un montant de remise, mais pas les deux';
            $context->buildViolation($message)
                    ->atPath('percent')
                    ->addViolation();
        }
    }
}
