<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Cart\Model;

use CpCreation\VitiCore\Behavior\HasDisable;
use CpCreation\VitiCore\Behavior\HasTimestamp;
use CpCreation\VitiCore\Behavior\Impl\Disable;
use CpCreation\VitiCore\Behavior\Impl\Timestamp;
use Doctrine\ORM\Mapping as ORM;
use Ramsey\Uuid\Uuid;
use Symfony\Component\Validator\Constraints as Assert;
use Symfony\Component\Validator\Context\ExecutionContextInterface;

/**
 * Class CartCheckout
 *
 * @ORM\Entity(repositoryClass="CpCreation\VitiCore\Cart\Repository\CartCheckoutRepository")
 * @ORM\Table(name="cart_cart_checkout")
 * @ORM\HasLifecycleCallbacks()
 */
class CartCheckout implements HasTimestamp, HasDisable
{
    use Timestamp, Disable;

    /**
     * @ORM\Id()
     * @ORM\Column(type="uuid")
     * @ORM\GeneratedValue(strategy="NONE")
     *
     * @var Uuid
     */
    private $id;

    /**
     * @ORM\Column(type="datetime", nullable=true)
     *
     * @var \DateTimeInterface
     */
    private $payedAt;

    /**
     * @ORM\Column(type="string", nullable=true)
     *
     * @var string
     */
    private $payedToken;

    /**
     * @var PaymentMethod
     *
     * @ORM\ManyToOne(targetEntity="CpCreation\VitiCore\Cart\Model\PaymentMethod")
     * @ORM\JoinColumn(name="payment_method_id", referencedColumnName="id")
     * @Assert\NotBlank(groups={"checkout"})
     */
    private $paymentMethod;

    /**
     * @ORM\Column(type="boolean", options={"default" : 1})
     *
     * @Assert\IsTrue()
     *
     * @var boolean
     */
    private $cgv;

    /**
     * @ORM\Column(type="boolean", options={"default" : 1})
     *
     * @var boolean
     */
    private $delivery;

    /**
     * @ORM\Column(type="string")
     *
     * @var string
     */
    private $locale = 'fr';

    /**
     * @ORM\Column(type="json", nullable=true)
     *
     * @var array
     */
    private $response;

    /**
     * @var boolean
     *
     * @ORM\Column(type="boolean")
     */
    private $colissimoCarrier;

    /**
     * @var string | null
     *
     * @ORM\Column(type="string", nullable=true)
     *
     */
    private $colissimoCode;

    /**
     * @var boolean
     *
     * @ORM\Column(type="boolean")
     */
    private $privateCarrier;

    /**
     * @var Cart
     *
     * @ORM\OneToOne(targetEntity="Cart", inversedBy="cartCheckout")
     * @ORM\JoinColumn(name="cart_cart_id", referencedColumnName="id", onDelete="CASCADE")
     */
    private $cart;

    public function __construct()
    {
        $this->id               = Uuid::uuid4();
        $this->delivery         = false;
        $this->colissimoCarrier = false;
        $this->privateCarrier   = false;
        $this->cgv              = false;
        $this->disabled         = false;

    }

    /**
     * @return Uuid
     */
    public function getId(): Uuid
    {
        return $this->id;
    }

    /**
     * @return \DateTimeInterface
     */
    public function getPayedAt(): ?\DateTimeInterface
    {
        return $this->payedAt;
    }

    /**
     * @param \DateTimeInterface $payedAt
     */
    public function setPayedAt(\DateTimeInterface $payedAt = null): void
    {
        $this->payedAt = $payedAt;
    }

    /**
     * @return string
     */
    public function getPayedToken(): ?string
    {
        return $this->payedToken;
    }

    /**
     * @param string $payedToken
     */
    public function setPayedToken(string $payedToken): void
    {
        $this->payedToken = $payedToken;
    }

    /**
     * @return PaymentMethod
     */
    public function getPaymentMethod(): ?PaymentMethod
    {
        return $this->paymentMethod;
    }

    /**
     * @param PaymentMethod $paymentMethod
     */
    public function setPaymentMethod(PaymentMethod $paymentMethod): void
    {
        $this->paymentMethod = $paymentMethod;
    }

    /**
     * @return bool
     */
    public function isCgv(): ?bool
    {
        return $this->cgv;
    }

    /**
     * @param bool $cgv
     */
    public function setCgv(bool $cgv): void
    {
        $this->cgv = $cgv;
    }

    /**
     * @return bool
     */
    public function isDelivery(): bool
    {
        return $this->delivery;
    }

    /**
     * @param bool $delivery
     */
    public function setDelivery(bool $delivery)
    {
        $this->delivery = $delivery;
    }

    /**
     * @return string
     */
    public function getLocale(): string
    {
        return $this->locale;
    }

    /**
     * @param string $locale
     */
    public function setLocale(string $locale)
    {
        $this->locale = $locale;
    }

    /**
     * @return array
     */
    public function getResponse(): ?array
    {
        return $this->response;
    }

    /**
     * @param array $response
     */
    public function setResponse(array $response = null)
    {
        $this->response = $response;
    }

    /**
     * @return bool
     */
    public function isColissimoCarrier(): bool
    {
        return $this->colissimoCarrier;
    }

    /**
     * @param bool $colissimoCarrier
     */
    public function setColissimoCarrier(bool $colissimoCarrier): void
    {
        $this->colissimoCarrier = $colissimoCarrier;
        if ($this->colissimoCarrier) {
            $this->setDelivery(true);
        }
    }

    /**
     * @return null|string
     */
    public function getColissimoCode(): ?string
    {
        return $this->colissimoCode;
    }

    /**
     * @param null|string $colissimoCode
     */
    public function setColissimoCode(?string $colissimoCode): void
    {
        $this->colissimoCode = $colissimoCode;
    }

    /**
     * @return bool
     */
    public function isPrivateCarrier(): bool
    {
        return $this->privateCarrier;
    }

    /**
     * @param bool $privateCarrier
     */
    public function setPrivateCarrier(bool $privateCarrier): void
    {
        $this->privateCarrier = $privateCarrier;
        if ($this->privateCarrier) {
            $this->setDelivery(true);
        }
    }

    /**
     * @return Cart
     */
    public function getCart(): Cart
    {
        return $this->cart;
    }

    /**
     * @param Cart $cart
     */
    public function setCart(Cart $cart): void
    {
        $this->cart = $cart;
    }

    /**
     * @param \Symfony\Component\Validator\Context\ExecutionContextInterface $context
     * @Assert\Callback(groups={"admin"})
     */
    public function validate(ExecutionContextInterface $context)
    {

        if (!$this->isPrivateCarrier() && !$this->isColissimoCarrier()) {
            $message = 'Veuillez choisir un choix de livraison';
            $context->buildViolation($message)
                    ->atPath("privateCarrier")
                    ->addViolation();
            $context->buildViolation($message)
                    ->atPath("colissimoCarrier")
                    ->addViolation();
        }
    }
}
