<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Bridge\Symfony\Bundle\Utils;

use CpCreation\VitiCore\Bridge\Symfony\Bundle\Filesystem\OrderFilesystem;
use CpCreation\VitiCore\Cart\Model\Cart;
use CpCreation\VitiCore\Contact\Model\Contact;
use CpCreation\VitiCore\Contact\Repository\ContactRepository;
use Symfony\Contracts\Translation\TranslatorInterface;

class Mailer
{
    private $pdf;

    private $mailer;

    private $filesystem;

    private $translator;

    private $emailToMaintenance;

    private $contactRepository;

    public function __construct(Pdf $pdf,
        \Swift_Mailer $mailer,
        OrderFilesystem $filesystem,
        TranslatorInterface $translator,
        ContactRepository $contactRepository,
        $emailToMaintenance
    ) {
        $this->pdf                = $pdf;
        $this->mailer             = $mailer;
        $this->filesystem         = $filesystem;
        $this->translator         = $translator;
        $this->emailToMaintenance = $emailToMaintenance;
        $this->contactRepository  = $contactRepository;
    }

    /**
     * @param Contact $contact
     * @param Cart    $cart
     * @throws \Twig\Error\LoaderError
     * @throws \Twig\Error\RuntimeError
     * @throws \Twig\Error\SyntaxError
     */
    public function sendOrder(Contact $contact, Cart $cart)
    {

        $path            = $this->pdf->createPdfOrder($contact, $cart);
        $messageCustomer = (new \Swift_Message($this->translator->trans('order.mail.subject_mail', ['%company%' => $contact->getCompany()])))
            ->setFrom($contact->getEmailFrom(), $contact->getCompany())
            ->setReplyTo($contact->getEmailToMessage())
            ->setTo($cart->getInvoiceAddress()->getEmail())
            ->attach(
                \Swift_Attachment::fromPath('order/'.$path, 'application/pdf')
            )
            ->setBody(

                nl2br($this->translator->trans('order.mail.text_mail', ['%company%' => $contact->getCompany()])),
                'text/html'
            )
            ->addPart(
                nl2br($this->translator->trans('order.mail.text_mail_plaintext', ['%company%' => $contact->getCompany()])),
                'text/plain'
            );

        $this->mailer->send($messageCustomer);

        $path    = $this->pdf->createPdfOrderAdmin($contact, $cart, 'fr');
        $message = (new \Swift_Message('Un nouveau bon de commande est arrivé ! de '.$cart->getInvoiceAddress()->getLastname().' '.$cart->getInvoiceAddress()->getFirstname()))
            ->setFrom($contact->getEmailFrom(), $contact->getCompany())
            ->setReplyTo($cart->getInvoiceAddress()->getEmail())
            ->setTo($contact->getEmailToShop())
            ->attach(
                \Swift_Attachment::fromPath('order/'.$path, 'application/pdf')
            )
            ->setBody(
                '
                Vous trouverez le bon de commande en pièce jointe de cet email.</br>
                '.$contact->getCompany(),
                'text/html'
            )
            ->addPart(
                '
                Vous trouverez le bon de commande en pièce jointe de cet email.
                '.$contact->getCompany(), 'text/plain'
            );

        $this->mailer->send($message);
    }

    /**
     * @param Contact $contact
     * @param Cart    $cart
     * @throws \Twig\Error\LoaderError
     * @throws \Twig\Error\RuntimeError
     * @throws \Twig\Error\SyntaxError
     */
    public function senDelivery(Contact $contact, Cart $cart)
    {
        $path = $this->pdf->createPdfOrder($contact, $cart);
        if ($cart->getCartCheckout()->isColissimoCarrier()) {
            if ($cart->getCartCheckout()->getColissimoCode()) {
                $body = $this->translator->trans(
                    'delivery.mail.colissimo_with_code', [
                    '%company%' => $contact->getCompany(),
                    '%ref%'     => $cart->getToken(),
                    '%code%'    => $cart->getCartCheckout()->getColissimoCode(),
                ], null, $cart->getCartCheckout()->getLocale()
                );
            } else {
                $body = $this->translator->trans(
                    'delivery.mail.colissimo', [
                    '%company%' => $contact->getCompany(),
                    '%ref%'     => $cart->getToken(),
                ], null, $cart->getCartCheckout()->getLocale()
                );
            }
        } else {
            $body = $this->translator->trans(
                'delivery.mail.private', [
                '%company%' => $contact->getCompany(),
                '%ref%'     => $cart->getToken(),
            ], null, $cart->getCartCheckout()->getLocale()
            );
        }
        $messageCustomer = (new \Swift_Message(
            $this->translator->trans(
                'delivery.mail.subject_mail', [
                '%company%' => $contact->getCompany(),
                '%ref%'     => $cart->getToken(),
            ], null, $cart->getCartCheckout()->getLocale()
            )
        ))
            ->setFrom($contact->getEmailFrom(), $contact->getCompany())
            ->setReplyTo($contact->getEmailToMessage())
            ->setTo($cart->getInvoiceAddress()->getEmail())
            ->attach(
                \Swift_Attachment::fromPath('order/'.$path, 'application/pdf')
            )
            ->setBody(
                nl2br(
                    $body
                ),
                'text/html'
            )
            ->addPart(
                strip_tags(
                    $body
                ),
                'text/plain'
            );

        $this->mailer->send($messageCustomer);
    }

    /**
     * @param $exception
     * @throws \Doctrine\ORM\NonUniqueResultException
     */
    public function sendErrorConnexionApiConnectSocialInstagram($exception)
    {

        /** @var Contact $contact */
        $contact = $this->contactRepository->findOne();
        $message = (new \Swift_Message('Erreur Connexion à l\'api Connect Social CPCREATION '.$contact->getCompany()))
            ->setFrom($contact->getEmailFrom(), $contact->getCompany())
            ->setTo($this->emailToMaintenance)
            ->setBody(
                $exception.' '.$contact->getCompany()
                ,
                'text/html'
            )
            ->addPart(

                $exception.' '.$contact->getCompany(), 'text/plain'
            );

        $this->mailer->send($message);
    }

    /**
     * @param $exception
     * @throws \Doctrine\ORM\NonUniqueResultException
     */
    public function sendErrorInstagram($exception)
    {

        /** @var Contact $contact */
        $contact = $this->contactRepository->findOne();
        $message = (new \Swift_Message('Erreur connexion à l\'api display Instagram '.$contact->getCompany()))
            ->setFrom($contact->getEmailFrom(), $contact->getCompany())
            ->setTo($this->emailToMaintenance)
            ->setBody(
                $exception.' '.$contact->getCompany()
                ,
                'text/html'
            )
            ->addPart(

                $exception.' '.$contact->getCompany(), 'text/plain'
            );

        $this->mailer->send($message);
    }

    /**
     * @param $exception
     * @throws \Doctrine\ORM\NonUniqueResultException
     */
    public function sendErrorFacebookInstagram($exception)
    {

        /** @var Contact $contact */
        $contact = $this->contactRepository->findOne();
        $message = (new \Swift_Message('Erreur connexion à l\'api Facebook pour Instagram '.$contact->getCompany()))
            ->setFrom($contact->getEmailFrom(), $contact->getCompany())
            ->setTo($this->emailToMaintenance)
            ->setBody(
                $exception.' '.$contact->getCompany()
                ,
                'text/html'
            )
            ->addPart(

                $exception.' '.$contact->getCompany(), 'text/plain'
            );

        $this->mailer->send($message);
    }

    /**
     * @param $exception
     * @throws \Doctrine\ORM\NonUniqueResultException
     */
    public function sendErrorFacebookFacebook($exception)
    {

        /** @var Contact $contact */
        $contact = $this->contactRepository->findOne();
        $message = (new \Swift_Message('Erreur connexion à l\'api Facebook pour Facebook '.$contact->getCompany()))
            ->setFrom($contact->getEmailFrom(), $contact->getCompany())
            ->setTo($this->emailToMaintenance)
            ->setBody(
                $exception.' '.$contact->getCompany()
                ,
                'text/html'
            )
            ->addPart(

                $exception.' '.$contact->getCompany(), 'text/plain'
            );

        $this->mailer->send($message);
    }

    /**
     * @param $exception
     * @throws \Doctrine\ORM\NonUniqueResultException
     */
    public function sendErrorException($exception)
    {

        /** @var Contact $contact */
        $contact = $this->contactRepository->findOne();
        $message = (new \Swift_Message('Erreur '.$contact->getCompany()))
            ->setFrom($contact->getEmailFrom(), $contact->getCompany())
            ->setTo($this->emailToMaintenance)
            ->setBody(
                $exception.' '.$contact->getCompany()
                ,
                'text/html'
            )
            ->addPart(

                $exception.' '.$contact->getCompany(), 'text/plain'
            );

        $this->mailer->send($message);
    }
}