<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Bridge\Symfony\Bundle\Payment;

use CpCreation\VitiCore\Cart\Model\Cart;
use CpCreation\VitiCore\Cart\Model\Payment\Monetico;
use CpCreation\VitiCore\Cart\Model\PaymentMethod;
use CpCreation\VitiCore\Cart\Repository\PaymentMethodRepository;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Routing\RouterInterface;
use Symfony\Component\Serializer\SerializerInterface;

class MoneticoPayment
{
    const METHOD             = 'monetico';
    const TEST_URL           = 'https://p.monetico-services.com/test/paiement.cgi';
    const PROD_URL           = 'https://p.monetico-services.com/paiement.cgi';
    const DEVISE             = 'EUR';
    const OPTIONS            = '';
    const CTLHMAC            = 'V4.0.sha1.php--[CtlHmac%s%s]-%s';
    const CTLHMACSTR         = 'CtlHmac%s%s';
    const PHASE2BACK_RECEIPT = 'version=2\ncdr=%s';
    const PHASE2BACK_FIELDS  = '%s*%s*%s*%s*%s*%s*%s*%s*%s*%s*%s*%s*%s*%s*%s*%s*%s*%s*%s*%s*';
    const PHASE1GO_FIELDS    = '%s*%s*%s%s*%s*%s*%s*%s*%s*%s*%s*%s*%s*%s*%s*%s*%s*%s*%s*%s';

    const NBR_ECH       = '';
    const DATE_ECH_1    = '';
    const MONTANT_ECH_1 = '';
    const DATE_ECH_2    = '';
    const MONTANT_ECH_2 = '';
    const DATE_ECH_3    = '';
    const MONTANT_ECH_3 = '';
    const DATE_ECH_4    = '';
    const MONTANT_ECH_4 = '';

    private $paymentMethodRepository;
    private $serializer;
    private $router;
    private $paths;
    private $request;

    public function __construct(PaymentMethodRepository $paymentMethodRepository,
        SerializerInterface $serializer,
        RouterInterface $router,
        RequestStack $requestStack,
        $paths
    ) {
        $this->paymentMethodRepository = $paymentMethodRepository;
        $this->serializer              = $serializer;
        $this->router                  = $router;
        $this->request                 = $requestStack;
        foreach ($paths as $key => $path) {
            if ($path['name'] === static::METHOD) {
                $this->paths[$key] = $path;
            }
        }
    }

    public function createData(Cart $cart)
    {
        $paymentMethod = $this->paymentMethodRepository->findOneBy(['method' => static::METHOD]);

        /** @var Monetico $config */
        $config = $this->serializer->denormalize($paymentMethod->getConfiguration(), Monetico::class);

        if ($config->isProduction()) {
            $data['URL'] = static::PROD_URL;
        } else {
            $data['URL'] = static::TEST_URL;
        }

        $total                  = $cart->getTotal() / 100;
        $data['version']        = $config->getVersion();
        $data['TPE']            = $config->getEptNumber();
        $data['date']           = date("d/m/Y:H:i:s");
        $data['text-libre']     = '';
        $data['montant']        = $total.static::DEVISE;
        $data['devise']         = static::DEVISE;
        $data['reference']      = strtoupper($cart->getToken());
        $data['url_retour']     = $this->request->getCurrentRequest()->getUriForPath($this->router->generate($this->paths['monetico']['canceled'], [], true));
        $data['url_retour_ok']  = $this->request->getCurrentRequest()->getUriForPath($this->router->generate($this->paths['monetico']['done'], [], true));
        $data['url_retour_err'] = $this->request->getCurrentRequest()->getUriForPath($this->router->generate($this->paths['monetico']['refused'], [], true));
        $data['lgue']           = strtoupper($this->request->getCurrentRequest()->getLocale());
        $data['societe']        = $config->getCompanyCode();
        $data['mail']           = $cart->getInvoiceAddress()->getEmail();
        $data['nbrech']         = static::NBR_ECH;
        $data['dateech1']       = static::DATE_ECH_1;
        $data['montantech1']    = static::MONTANT_ECH_1;
        $data['dateech2']       = static::DATE_ECH_2;
        $data['montantech2']    = static::MONTANT_ECH_2;
        $data['dateech3']       = static::DATE_ECH_3;
        $data['montantech3']    = static::MONTANT_ECH_3;
        $data['dateech4']       = static::DATE_ECH_4;
        $data['montantech4']    = static::MONTANT_ECH_4;

        //Version 3 Not work
        /*$dataCalcul = $data;
        unset($dataCalcul['URL']);
        unset($dataCalcul['devise']);
        ksort($dataCalcul);
        array_walk($dataCalcul, function (&$a, $b) { $a = "$b=$a"; });
        $msgGo = implode('*', $dataCalcul);*/

        //Version 2
        $msg         = sprintf(
            static::PHASE1GO_FIELDS,
            $data['TPE'],
            $data['date'],
            $total,
            $data['devise'],
            $data['reference'],
            $data['text-libre'],
            $data['version'],
            $data['lgue'],
            $data['societe'],
            $data['mail'],
            $data['nbrech'],
            $data['dateech1'],
            $data['montantech1'],
            $data['dateech2'],
            $data['montantech2'],
            $data['dateech3'],
            $data['montantech3'],
            $data['dateech4'],
            $data['montantech4'],
            static::OPTIONS


        );
        $data['MAC'] = $this->computeHmac($msg, $config->getHmacKey());

        return $data;
    }

    private function _getUsableKey($hmacKey)
    {
        $hexStrKey = substr($hmacKey, 0, 38);
        $hexFinal  = "".substr($hmacKey, 38, 2)."00";

        $cca0 = ord($hexFinal);

        if ($cca0 > 70 && $cca0 < 97) {
            $hexStrKey .= chr($cca0 - 23).substr($hexFinal, 1, 1);
        } else {
            if (substr($hexFinal, 1, 1) == "M") {
                $hexStrKey .= substr($hexFinal, 0, 1)."0";
            } else {
                $hexStrKey .= substr($hexFinal, 0, 2);
            }
        }

        return pack("H*", $hexStrKey);
    }

    public function computeHmac($sData, $hmac)
    {

        return strtolower(hash_hmac("sha1", $sData, $this->_getUsableKey($hmac)));
    }

    public function valid($response = null)
    {

        if ($response) {
            $paymentMethod = $this->paymentMethodRepository->findOneBy(['method' => static::METHOD]);
            /** @var Monetico $config */
            $config = $this->serializer->denormalize($paymentMethod->getConfiguration(), Monetico::class);

            // Version 3 Not Work
            /* $mac=$response['MAC'];
             unset($response['MAC']);
             $response['version'] = $config->getVersion();
             ksort($response);
             array_walk($response, function(&$a, $b) { $a = "$b=$a"; });
             $msgBack = implode( '*', $response)*/;

            //version 2
            $mac = $response['MAC'];
            if (!isset($response['texte-libre'])) {
                $response['texte-libre'] = '';
            }

            if (!isset($response['motifrefus'])) {
                $response['motifrefus'] = '';
            }

            $msgBack = sprintf(
                static::PHASE2BACK_FIELDS,
                $response['TPE'],
                $response['date'],
                $response['montant'],
                $response['reference'],
                $response['texte-libre'],
                $config->getVersion(),
                $response['code-retour'],
                $response['cvx'],
                $response['vld'],
                $response['brand'],
                $response['status3ds'],
                $response['numauto'],
                $response['motifrefus'],
                $response['originecb'],
                $response['bincb'],
                $response['hpancb'],
                $response['ipclient'],
                $response['originetr'],
                $response['veres'],
                $response['pares']
            );

            if ($this->computeHmac($msgBack, $config->getHmacKey()) == strtolower($mac)) {

                return true;
            }
        }

        return false;
    }
}