<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Site;

use CpCreation\VitiCore\Bridge\Symfony\Bundle\Form\Site\CartType;
use CpCreation\VitiCore\Bridge\Symfony\Bundle\Form\Site\CheckCodePriceType;
use CpCreation\VitiCore\Bridge\Symfony\Bundle\Payment\MoneticoPayment;
use CpCreation\VitiCore\Bridge\Symfony\Bundle\Utils\CalculDegressive;
use CpCreation\VitiCore\Bridge\Symfony\Bundle\Utils\CalculShipping;
use CpCreation\VitiCore\Bridge\Symfony\Bundle\Utils\Mailer;
use CpCreation\VitiCore\Cart\Model\Cart;
use CpCreation\VitiCore\Cart\Model\CartItem;
use CpCreation\VitiCore\Cart\Model\InvoiceAddress;
use CpCreation\VitiCore\Cart\Model\Promo;
use CpCreation\VitiCore\Cart\Repository\CartRepository;
use CpCreation\VitiCore\Cart\Repository\PromoRepository;
use CpCreation\VitiCore\Contact\Repository\ContactRepository;
use CpCreation\VitiCore\Product\Repository\ProductPriceCodeRepository;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Cookie;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\Session\SessionInterface;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;
use Symfony\Component\Routing\Annotation\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;
use Symfony\Component\Routing\RouterInterface;
use Symfony\Contracts\Translation\TranslatorInterface;
use VatValidation\Validation;

class CartController extends AbstractController
{
    private $calculShipping;
    private $calculDegressive;
    private $promoRepository;
    private $productPriceCodeRepository;
    private $cartRepository;
    private $contactRepository;
    private $router;
    private $translator;
    private $mailer;
    private $moneticoPayment;
    private $em;
    private $session;

    public function __construct(CalculShipping $calculShipping,
        CalculDegressive $calculDegressive,
        PromoRepository $promoRepository,
        ProductPriceCodeRepository $productPriceCodeRepository,
        CartRepository $cartRepository,
        ContactRepository $contactRepository,
        RouterInterface $router,
        TranslatorInterface $translator,
        Mailer $mailer,
        MoneticoPayment $moneticoPayment,
        EntityManagerInterface $em,
        SessionInterface $session
    ) {
        $this->calculShipping             = $calculShipping;
        $this->calculDegressive           = $calculDegressive;
        $this->promoRepository            = $promoRepository;
        $this->productPriceCodeRepository = $productPriceCodeRepository;
        $this->cartRepository             = $cartRepository;
        $this->contactRepository          = $contactRepository;
        $this->router                     = $router;
        $this->translator                 = $translator;
        $this->mailer                     = $mailer;
        $this->moneticoPayment            = $moneticoPayment;
        $this->em                         = $em;
        $this->session                    = $session;
    }

    /**
     * @param Request $request
     *
     * @Route("/request/calcul_shipping", options={"expose"=true}, name="calcul_shipping")
     * @return JsonResponse
     */
    public
    function shippingCalculAction(Request $request
    ) {
        $cart = $this->cartRepository->findOneBy(['id' => $this->session->get('cart')]);
        if (!$cart) {
            $cart = new Cart();
            $this->session->set('cart', $cart->getId()->toString());
        }
        $form = $this->createForm(CartType::class, $cart);
        $form->handleRequest($request);
        $items = $cart->getItems();
        foreach ($items as $item) {
            $variant = $item->getVariant();
            $item->setCart($cart);
            if ($variant->getPriceDiscount()) {
                $item->setPrice($variant->getPriceDiscount());
            } else {
                $item->setPrice($variant->getPrice());
            }
        }
        $cart->setUpdatedAt(new \DateTime());
        $this->cartRepository->save($cart);
        try {
            $shipping = $this->calculShipping->shippingTotalPrice($cart);

            return new JsonResponse($shipping['amount'], 200);
        } catch (\Exception $e) {

            return new JsonResponse($e->getMessage(), 400);
        }
    }

    /**
     * @param Request $request
     *
     * @Route("/request/promo_code", options={"expose"=true}, name="promo_code")
     *
     * @return JsonResponse
     * @throws \Doctrine\ORM\NonUniqueResultException
     */
    public
    function codePromoAction(Request $request
    ) {
        $locale = $request->getLocale();
        $code   = $request->get('code');
        $form   = $this->createForm(CartType::class);
        $form->handleRequest($request);
        $quantityAllAbsolu = 0;
        /** @var Cart $cart */
        $cart = $form->getData();
        /** @var CartItem $item */
        foreach ($cart->getItems() as $item) {
            $quantityAllAbsolu = $quantityAllAbsolu + ($item->getQuantity() * $item->getVariant()->getType()->getEquivalency());
        }

        /** @var Promo $promo */
        $promo = $this->promoRepository->findPromoActivWithCode($code);
        if ($promo) {
            if (!$promo->getMaxCount() || (($promo->getCount() < $promo->getMaxCount()))) {

                if ($promo->getMinEquivalency() && $quantityAllAbsolu < $promo->getMinEquivalency()) {
                    return new JsonResponse(
                        [
                            'value'       => null,
                            'valueAmount' => null,
                            'message'     => $this->translator->trans('shop.promo_code.qty_error', ['%qty%' => $promo->getMinEquivalency()], null, $locale),
                        ], 400
                    );
                }

                return new JsonResponse(
                    [
                        'valuePercent' => $promo->getPercent(),
                        'valueAmount'  => $promo->getAmount(),
                        'message'      => $this->translator->trans('shop.promo_code.success', [], null, $locale),
                    ], 200
                );
            } else {
                return new JsonResponse(
                    [
                        'value'       => null,
                        'valueAmount' => null,
                        'message'     => $this->translator->trans('shop.promo_code.count_error', [], null, $locale),
                    ], 400
                );
            }

        } else {
            return new JsonResponse(
                [
                    'value'       => null,
                    'valueAmount' => null,
                    'message'     => $this->translator->trans('shop.promo_code.error', [], null, $locale),
                ], 400
            );
        }
    }

    /**
     * @param Request $request
     *
     * @Route("/request/price_promo", options={"expose"=true}, name="promo_price")
     *
     * @return JsonResponse
     * @throws \Doctrine\ORM\NonUniqueResultException
     * @throws \Exception
     */
    public
    function pricePromoAction(Request $request
    ) {
        $form = $this->createForm(CartType::class);
        $form->handleRequest($request);
        $cart       = $form->getData();
        $degressive = $this->calculDegressive->calcul($cart);

        if ($degressive) {
            return new JsonResponse($degressive, 200);
        }

        return new JsonResponse(null, 400);
    }

    /**
     * @Template("@CpCreationVitiCore/site/_formCodePrice.html.twig")
     * @Route(path="/request/check-code-price", name="price_code")
     * @param Request $request
     *
     * @return array|\Symfony\Component\HttpFoundation\RedirectResponse
     */
    public
    function codePriceGroupAction(Request $request
    ) {
        $locale = $request->getLocale();
        $form   = $this->createForm(CheckCodePriceType::class);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $data   = $form->getData();
            $code   = str_replace(' ', '', $data['code']);
            $result = $this->productPriceCodeRepository->findOneBy(['code' => $code]);

            if ($result) {
                $response = new RedirectResponse($this->router->generate('shop', ['_locale' => $locale]));
                $response->headers->setCookie(new Cookie('PRICE', $result->getId(), time() + (1 * 24 * 60 * 60), '/', null, false, false));
                $this->addFlash(
                    'success', $this->translator->trans(
                    'shop.form.partner.success', [
                    '%code%' => $code,
                ], null, $locale
                )
                );

                return $response;
            }
            $vatValidation = new Validation($code);
            try {
                $vatValidation->isValid();
                $result = $this->productPriceCodeRepository->findOneBy(['viewHT' => true]);
                if (!$result) {
                    $this->addFlash('error', $this->translator->trans('shop.form.partner.error_empty', ['_locale' => $locale]));

                    return $this->redirectToRoute('shop', ['_locale' => $locale]);
                }
            } catch (\Exception $e) {
                $this->addFlash(
                    'error', $this->translator->trans(
                    'shop.form.partner.error_exist', [
                    '%code%' => $code,
                ], null, $locale
                )
                );

                return $this->redirectToRoute('shop', ['_locale' => $locale]);
            }

            try {
                $vatInfos = $vatValidation->checkVat();
                if ($vatInfos->valid) {
                    $cart           = new Cart();
                    $invoiceAddress = new InvoiceAddress();
                    $invoiceAddress->setCompany($vatInfos->name);
                    $invoiceAddress->setCountry($vatInfos->countryCode);

                    $cart->setInvoiceAddress($invoiceAddress);
                    $this->getDoctrine()->getManager()->persist($cart);
                    $this->getDoctrine()->getManager()->flush();
                    $this->get('session')->set('cart', $cart->getId());
                    $response = new RedirectResponse($this->router->generate('shop', ['_locale' => $locale]));

                    $response->headers->setCookie(new Cookie('PRICE', $result->getId(), time() + (1 * 24 * 60 * 60), '/', null, false, false));
                    $this->addFlash(
                        'success', $this->translator->trans(
                        'shop.form.partner.success', [
                        '%code%' => $code,
                    ], null, $locale
                    )
                    );

                    return $response;
                }
                $this->addFlash(
                    'error', $this->translator->trans(
                    'shop.form.partner.error_exist', [
                    '%code%' => $code,
                ], null, $locale
                )
                );

                return $this->redirectToRoute('shop');
            } catch (\Exception $e) {
                $this->addFlash(
                    'error', $this->translator->trans(
                    'shop.form.partner.error_exist', [
                    '%code%' => $code,
                ], null, $locale
                )
                );

                return $this->redirectToRoute('shop', ['_locale' => $locale]);
            }
        }

        return [
            'form' => $form->createView(),
        ];
    }

    /**
     * @Route(path="/request/check-code-price/delete", name="price_code_delete")
     *
     * @return array|\Symfony\Component\HttpFoundation\RedirectResponse
     */
    public
    function deleteCodePriceGroupAction(Request $request
    ) {
        $locale   = $request->getLocale();
        $response = new RedirectResponse($this->router->generate('shop', ['_locale' => $locale]));
        $this->get('session')->set('cart', null);
        $response->headers->clearCookie('PRICE');

        return $response;
    }

    /**
     * @param Request $request
     * @Route("/request/etransactions/responseAt", name="etransactions_responseAt")
     *
     * @return Response
     * @throws \Doctrine\ORM\NonUniqueResultException
     * @throws \Twig\Error\LoaderError
     * @throws \Twig\Error\RuntimeError
     * @throws \Twig\Error\SyntaxError
     */
    public
    function EtransactionsResponseAt(Request $request
    ) {

        if ($request->request->get('Erreur')) {

            $code = $request->request->get('Erreur', null);

        } else {
            $code = $request->query->get('Erreur', null);
        }

        if ($request->request->get('Ref')) {

            $ref = $request->request->get('Ref', null);

        } else {
            $ref = $request->query->get('Ref', null);
        }

        if ($request->request->all()) {

            $response = $request->request->all();

        } else {
            $response = $request->query->all();
        }
        if (!$ref || !$code) {
            throw new BadRequestHttpException();
        }
        /** Accès au variant désactivé, si paiement effectué après la désactivation d'un variant  */
        $this->em->getFilters()->disable("soft_deletable");
        /** @var Cart $cart */
        $cart = $this->cartRepository->findOneBy(['token' => $ref]);
        if ($cart) {
            $checkout = $cart->getCartCheckout();
            $checkout->setResponse($response);
            if ($code === '00000') {
                $checkout->setPayedAt(new \DateTime('now'));
                $this->getDoctrine()->getManager()->persist($checkout);
                $this->getDoctrine()->getManager()->flush();
                $contact = $this->contactRepository->findOne();
                $this->mailer->sendOrder($contact, $cart);
            } else {
                $this->getDoctrine()->getManager()->persist($checkout);
                $this->getDoctrine()->getManager()->flush();
            }
        }

        return new Response('', 200);

    }

    /**
     * @param Request $request
     * @Route("/request/monetico/responseAt", name="monetico_responseAt")
     *
     * @return Response
     * @throws \Doctrine\ORM\NonUniqueResultException
     * @throws \Twig\Error\LoaderError
     * @throws \Twig\Error\RuntimeError
     * @throws \Twig\Error\SyntaxError
     */
    public
    function MoneticoResponseAt(Request $request
    ) {
        if ($request->request->get('code-retour')) {

            $code = $request->request->get('code-retour', null);

        } else {
            $code = $request->query->get('code-retour', null);
        }

        if ($request->request->get('reference')) {

            $ref = $request->request->get('reference', null);

        } else {
            $ref = $request->query->get('reference', null);
        }

        if ($request->request->all()) {

            $response = $request->request->all();

        } else {
            $response = $request->query->all();
        }

        if (!$ref || !$code) {
            throw new BadRequestHttpException();
        }

        if ($this->moneticoPayment->valid($response)) {
            /** Accès au variant désactivé, si paiement effectué après la désactivation d'un variant  */
            $this->em->getFilters()->disable("soft_deletable");
            /** @var Cart $cart */
            $cart = $this->cartRepository->findOneBy(['token' => $ref]);
            if ($cart) {
                $checkout = $cart->getCartCheckout();
                $checkout->setResponse($response);
                if ($code === 'payetest' || $code === 'paiement') {
                    $checkout->setPayedAt(new \DateTime('now'));
                    $this->getDoctrine()->getManager()->persist($checkout);
                    $this->getDoctrine()->getManager()->flush();
                    $contact = $this->contactRepository->findOne();
                    $this->mailer->sendOrder($contact, $cart);
                } else {
                    $this->getDoctrine()->getManager()->persist($checkout);
                    $this->getDoctrine()->getManager()->flush();
                }

                return new Response('version=2'."\n".'cdr=0', 200);
            }

            return new Response('version=2'."\n".'cdr=1'."\n", 200);

        } else {
            return new Response('version=2'."\n".'cdr=1'."\n", 200);
        }
    }
}
