<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Admin;

use CpCreation\VitiCore\Bridge\Symfony\Bundle\Form\Admin\SeasonType;
use CpCreation\VitiCore\Season\Model\Season;
use CpCreation\VitiCore\Season\Repository\SeasonRepository;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\IsGranted;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Annotation\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;

/**
 * @IsGranted({"ROLE_ADMIN"})
 * Class SeasonController
 * @package CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Admin
 * @Route("/season")
 */
class SeasonController extends AbstractController
{
    private $seasonRepository;

    public function __construct(SeasonRepository $seasonRepository)
    {
        $this->seasonRepository =  $seasonRepository;
    }

    /**
     * @Route("", name="admin_season")
     * @Template("@CpCreationVitiCore/admin/season/index.html.twig")
     */
    public function indexAction()
    {
        $seasons = $this->seasonRepository->findBy([], ['position' => 'ASC']);

        return [
            "items" => $seasons
        ];
    }

    /**
     * @param Request $request
     * @Route("/new", name="admin_season_new")
     * @Template("@CpCreationVitiCore/admin/season/new.html.twig")
     *
     * @return array|\Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function newAction(Request $request)
    {
        $form = $this->createForm(SeasonType::class);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            /** @var Season $data */
            $data = $form->getData();
            $this->seasonRepository->save($data);
            $this->addFlash('alert-success', 'La saison '.$data->getTitle().' a été ajouté avec succès');

            return $this->redirectToRoute('admin_season');
        }

        return [
            "form" => $form->createView(),
        ];
    }

    /**
     * @param Season $season
     * @param Request $request
     * @Route("/{id}/edit", name="admin_season_edit")
     * @Template("@CpCreationVitiCore/admin/season/edit.html.twig")
     *
     * @return array|\Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function editAction(Season $season, Request $request){
        $form = $this->createForm(SeasonType::class, $season);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            /** @var Season $data */
            $data = $form->getData();
            $data->setUpdatedAt(new \DateTime()); // Forcer Update de doctrine
            $this->seasonRepository->save($data);
            $this->addFlash('alert-success', 'La saison '.$season->getTitle().' a été modifié avec succès');

            return $this->redirectToRoute('admin_season');
        }

        return [
            "item" => $season,
            "form" => $form->createView(),
        ];
    }

    /**
     * @param Season $season
     * @Route("/{id}/delete", name="admin_season_delete")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function deleteAction(Season $season)
    {
        $this->seasonRepository->delete($season);
        $this->addFlash('alert-success', 'La saison '.$season->getTitle().' a été supprimé avec succès.');

        return $this->redirectToRoute('admin_season');
    }

    /**
     * @param Season $season
     * @Route("/{id}/moveUp", name="admin_season_move_up")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function moveUpAction(Season $season)
    {
        $season->setPosition($season->getPosition() - 1);
        $this->seasonRepository->save($season);
        $this->addFlash('alert-success', 'La saison '.$season->getTitle().' a été remonté.');

        return $this->redirectToRoute('admin_season');
    }

    /**
     * @param Season $season
     * @Route("/{id}/moveDown", name="admin_season_move_down")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function moveDownAction(Season $season)
    {
        $season->setPosition($season->getPosition() + 1);
        $this->seasonRepository->save($season);
        $this->addFlash('alert-success', 'La saison '.$season->getTitle().' a été descendu.');

        return $this->redirectToRoute('admin_season');
    }
}