<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Admin;

use CpCreation\VitiCore\Bridge\Symfony\Bundle\Form\Admin\ProductType;
use CpCreation\VitiCore\Product\Model\Product;
use CpCreation\VitiCore\Product\Repository\ProductRepository;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\IsGranted;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\Routing\Annotation\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;
use Symfony\Component\HttpFoundation\Request;

/**
 * @IsGranted({"ROLE_ADMIN","ROLE_PRODUCT","ROLE_PRODUCT_PRICE"})
 * Class ProductController
 * @package CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Admin
 * @Route("/produits")
 */
class ProductController extends AbstractController
{
    private $productRepository;

    public function __construct(ProductRepository $productRepository)
    {
        $this->productRepository = $productRepository;
    }

    /**
     * @route("", name="admin_product")
     * @Template("@CpCreationVitiCore/admin/product/index.html.twig")
     */
    public function indexAction()
    {
        $products = $this->productRepository->findBy([], ['position' => 'ASC']);

        return [
            "items" => $products,
        ];
    }

    /**
     * @param Request $request
     * @Route("/new", name="admin_product_new")
     * @Template("@CpCreationVitiCore/admin/product/new.html.twig")
     *
     * @return array|\Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function newAction(Request $request)
    {
        $product = new Product();
        $form    = $this->createForm(ProductType::class, $product);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            /** @var Product $data */
            $data = $form->getData();
            $this->productRepository->save($data);
            $this->addFlash('alert-success', 'Le produit '.$data->getName().' a été ajouté avec succès');

            return $this->redirectToRoute('admin_product');

        }

        return [
            "form" => $form->createView(),
        ];
    }

    /**
     * @param Product $product
     * @param Request $request
     * @Route("/{id}/edit", name="admin_product_edit")
     * @Template("@CpCreationVitiCore/admin/product/edit.html.twig")
     *
     * @return array|\Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function editAction(Product $product, Request $request)
    {
        $form = $this->createForm(ProductType::class, $product);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            /** @var Product $data */
            $data = $form->getData();
            $data->setUpdatedAt(new \DateTime()); // Forcer Update de doctrine
            $this->productRepository->save($data);
            $this->addFlash('alert-success', 'Le produit '.$product->getName().' a été modifié avec succès');

            return $this->redirectToRoute('admin_product');

        }

        return [
            "item" => $product,
            "form" => $form->createView(),
        ];
    }

    /**
     * @param Product $product
     * @Route("/{id}/delete", name="admin_product_delete")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function deleteAction(Product $product)
    {
        $this->productRepository->delete($product);
        $this->addFlash('alert-success', 'Le produit '.$product->getName().' a été supprimé avec succès.');

        return $this->redirectToRoute('admin_product');
    }

    /**
     * @param Product $product
     * @Route("/{id}/disable", name="admin_product_disable")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function disableAction(Product $product)
    {
        $product->setDisabled(1);
        $this->productRepository->save($product);
        $this->addFlash('alert-success', 'Le produit '.$product->getName().' a été dépublié.');

        return $this->redirectToRoute('admin_product');
    }

    /**
     * @param Product $product
     * @Route("/{id}/enable", name="admin_product_enable")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function enableAction(Product $product)
    {
        $product->setDisabled(0);
        $this->productRepository->save($product);
        $this->addFlash('alert-success', 'Le produit '.$product->getName().' a été publié.');

        return $this->redirectToRoute('admin_product');
    }

    /**
     * @param Product $product
     * @Route("/{id}/moveUp", name="admin_product_move_up")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function moveUpAction(Product $product)
    {
        $product->setPosition($product->getPosition() - 1);
        $this->productRepository->save($product);
        $this->addFlash('alert-success', 'Le produit '.$product->getName().' a été remonté.');

        return $this->redirectToRoute('admin_product');
    }

    /**
     * @param Product $product
     * @Route("/{id}/moveDown", name="admin_product_move_down")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function moveDownCuveeAction(Product $product)
    {
        $product->setPosition($product->getPosition() + 1);
        $this->productRepository->save($product);
        $this->addFlash('alert-success', 'Le produit '.$product->getName().' a été descendu.');

        return $this->redirectToRoute('admin_product');
    }
}