<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Admin;

use CpCreation\VitiCore\Bridge\Symfony\Bundle\Form\Admin\ProductCategoryType;
use CpCreation\VitiCore\Product\Model\ProductCategory;
use CpCreation\VitiCore\Product\Repository\ProductCategoryRepository;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\IsGranted;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\Routing\Annotation\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;
use Symfony\Component\HttpFoundation\Request;

/**
 * @IsGranted({"ROLE_ADMIN"})
 * Class ProductCategoryController
 * @package CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Admin
 * @Route("/produit_categories")
 */
class ProductCategoryController extends AbstractController
{
    private $productCategoryRepository;

    public function __construct(ProductCategoryRepository $productCategoryRepository)
    {
        $this->productCategoryRepository = $productCategoryRepository;
    }

    /**
     * @route("", name="admin_product_category")
     * @Template("@CpCreationVitiCore/admin/product/category/index.html.twig")
     */
    public function indexAction()
    {
        $products = $this->productCategoryRepository->findBy([],["position" => "ASC"]);

        return [
            "items" => $products,
        ];
    }

    /**
     * @param Request $request
     * @Route("/new", name="admin_product_category_new")
     * @Template("@CpCreationVitiCore/admin/product/category/new.html.twig")
     *
     * @return array|\Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function newAction(Request $request)
    {

        $form = $this->createForm(ProductCategoryType::class);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            /** @var ProductCategory $data */
            $data = $form->getData();
            $this->productCategoryRepository->save($data);
            $this->addFlash('alert-success', 'La catégorie '.$data->getName().' a été ajouté avec succès');

            return $this->redirectToRoute('admin_product_category');

        }

        return [
            "form" => $form->createView(),
        ];
    }

    /**
     * @param ProductCategory $category
     * @param Request $request
     * @Route("/{id}/edit", name="admin_product_category_edit")
     * @Template("@CpCreationVitiCore/admin/product/category/edit.html.twig")
     *
     * @return array|\Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function editAction(ProductCategory $category, Request $request)
    {
        $form = $this->createForm(ProductCategoryType::class, $category);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            /** @var Category $data */
            $data = $form->getData();
            $this->productCategoryRepository->save($data);
            $this->addFlash('alert-success', 'La catégorie '.$category->getName().' a été modifié avec succès');

            return $this->redirectToRoute('admin_product_category');

        }

        return [
            "item" => $category,
            "form" => $form->createView(),
        ];
    }

    /**
     * @param ProductCategory $category
     * @Route("/{id}/moveUp", name="admin_product_category_move_up")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function moveUpAction(ProductCategory $category)
    {
        $category->setPosition($category->getPosition() - 1);
        $this->productCategoryRepository->save($category);
        $this->addFlash('alert-success', 'La catégorie '.$category->getName().' a été remonté.');

        return $this->redirectToRoute('admin_product_category');
    }

    /**
     * @param ProductCategory $category
     * @Route("/{id}/moveDown", name="admin_product_category_move_down")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function moveDownCuveeAction(ProductCategory $category)
    {
        $category->setPosition($category->getPosition() + 1);
        $this->productCategoryRepository->save($category);
        $this->addFlash('alert-success', 'La categorie '.$category->getName().' a été descendu.');

        return $this->redirectToRoute('admin_product_category');
    }

    /**
     * @param ProductCategory $category
     * @Route("/{id}/delete", name="admin_product_category_delete")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function deleteAction(ProductCategory $category)
    {
        $this->productCategoryRepository->delete($category);
        $this->addFlash('alert-success', 'La catégorie '.$category->getName().' a été supprimé avec succès.');

        return $this->redirectToRoute('admin_product_category');
    }
}