<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Shop\Model;

use CpCreation\VitiCore\Behavior\HasDisable;
use CpCreation\VitiCore\Behavior\Impl\Disable;
use CpCreation\VitiCore\Product\Model\ProductPriceGroup;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Ramsey\Uuid\Uuid;
use Ramsey\Uuid\UuidInterface;
use Symfony\Component\Intl\Countries;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * Class ShippingCountry
 *
 * @ORM\Entity(repositoryClass="CpCreation\VitiCore\Shop\Repository\ShippingCountryRepository")
 * @ORM\Table(name="shop_shipping_country")
 * @ORM\HasLifecycleCallbacks()
 */
class ShippingCountry implements HasDisable
{
    use Disable;

    /**
     * @ORM\Id()
     * @ORM\Column(type="uuid")
     * @ORM\GeneratedValue(strategy="NONE")
     *
     * @var UuidInterface
     */
    private $id;

    /**
     * @ORM\Column(type="string")
     * @Assert\NotBlank()
     *
     * @var string
     */
    private $code;

    /**
     * @ORM\Column(type="boolean")
     *
     * @var boolean
     */
    private $county;

    /**
     * @ORM\Column(type="integer", nullable=true)
     *
     * @var integer | null
     */
    private $minQty;

    /**
     * @ORM\Column(type="integer", nullable=true)
     *
     * @var integer | null
     */
    private $multiple;

    /**
     * @ORM\ManyToMany(targetEntity="ShippingZone", mappedBy="countries")
     *
     * @var Collection | ShippingZone[]
     */
    private $zones;

    /**
     * @ORM\ManyToMany(targetEntity="CpCreation\VitiCore\Product\Model\ProductPriceGroup")
     * @Assert\Count( min = 1 , minMessage="Vous devez ajouter au moins un groupe")
     * @var Collection<ProductPriceGroup>
     */
    private $groups;

    public function __construct()
    {
        $this->id     = Uuid::uuid4();
        $this->county = false;
        $this->disabled = true;
        $this->groups = new ArrayCollection();
        $this->zones  = new ArrayCollection();
    }

    /**
     * @return UuidInterface
     */
    public function getId(): UuidInterface
    {
        return $this->id;
    }

    /**
     * @return string
     */
    public function getCode(): ?string
    {
        return $this->code;
    }

    /**
     * @param string $code
     */
    public function setCode(string $code): void
    {
        $this->code = $code;
    }

    /**
     * @return bool
     */
    public function isCounty(): bool
    {
        return $this->county;
    }

    /**
     * @param bool $county
     */
    public function setCounty(bool $county): void
    {
        $this->county = $county;
    }

    /**
     * @return int|null
     */
    public function getMinQty(): ?int
    {
        return $this->minQty;
    }

    /**
     * @param int|null $minQty
     */
    public function setMinQty(?int $minQty): void
    {
        $this->minQty = $minQty;
    }

    /**
     * @return int|null
     */
    public function getMultiple(): ?int
    {
        return $this->multiple;
    }

    /**
     * @param int|null $multiple
     */
    public function setMultiple(?int $multiple): void
    {
        $this->multiple = $multiple;
    }

    /**
     * @return ShippingZone[]|Collection
     */
    public function getZones()
    {
        return $this->zones;
    }

    /**
     * @param ShippingZone $zone
     * @return $this
     */
    public function addZone(ShippingZone $zone)
    {
        if (!$this->zones->contains($zone)) {
            $this->zones[] = $zone;
        }

        return $this;
    }

    /**
     * @param ShippingZone $zone
     */
    public function removeZone(ShippingZone $zone)
    {
        if ($this->zones->contains($zone)) {
            $this->zones->removeElement($zone);
        }

    }

    /**
     * @param ProductPriceGroup $group
     * @return $this
     */
    public function addGroup(ProductPriceGroup $group)
    {
        $this->groups[] = $group;

        return $this;
    }

    /**
     * @param ProductPriceGroup $group
     */
    public function removeGroup(ProductPriceGroup $group)
    {
        $this->groups->removeElement($group);
    }

    /**
     * @return Collection<ProductPriceGroup>
     */
    public function getGroups(): Collection
    {
        return $this->groups;
    }

    public function getName()
    {
        if ($this->getCode())
        {
            return Countries::getName($this->getCode());
        }

        return null;
    }
}
