<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Bridge\Symfony\Bundle\Payment;

use CpCreation\VitiCore\Behavior\HasPayment;
use CpCreation\VitiCore\Bridge\Symfony\Bundle\Utils\Mailer;
use CpCreation\VitiCore\Cart\Model\Payment\Axepta;
use CpCreation\VitiCore\Cart\Repository\PaymentMethodRepository;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Routing\RouterInterface;
use Symfony\Component\Serializer\SerializerInterface;

class AxeptaPayment
{
    const METHOD = 'axepta';
    const URL    = 'https://paymentpage.axepta.bnpparibas/payssl.aspx';
    const DEVISE = 'EUR';

    private $paymentMethodRepository;
    private $serializer;
    private $router;
    private $paths;
    private $request;

    public function __construct(PaymentMethodRepository $paymentMethodRepository,
        SerializerInterface $serializer,
        RouterInterface $router,
        RequestStack $requestStack,
        Mailer $mailer,
        $paths
    ) {
        $this->paymentMethodRepository = $paymentMethodRepository;
        $this->serializer              = $serializer;
        $this->router                  = $router;
        $this->request                 = $requestStack;
        $this->mailer                  = $mailer;
        foreach ($paths as $key => $path) {
            if ($path['name'] === static::METHOD) {
                $this->paths[$key] = $path;
            }
        }
    }

    public function createData(HasPayment $payment)
    {
        $paymentMethod = $this->paymentMethodRepository->findOneBy(['method' => static::METHOD]);

        /** @var Axepta $config */
        $config = $this->serializer->denormalize($paymentMethod->getConfiguration(), Axepta::class);

        $hmac     = $config->getHmacKey();
        $blowfish = $config->getBlowfish();

        $data['MerchantID'] = $config->getMerchantId();
        $data['TransID']    = $payment->getToken();
        $data['Amount']     = $payment->getTotal();
        $data['Currency']   = static::DEVISE;
        $data['URLBack']    = $this->request->getCurrentRequest()->getUriForPath($this->router->generate($this->paths['axepta']['canceled'], [], true));
        $data['URLSuccess'] = $this->request->getCurrentRequest()->getUriForPath($this->router->generate($this->paths['axepta']['done'], [], true));
        $data['URLFailure'] = $this->request->getCurrentRequest()->getUriForPath($this->router->generate($this->paths['axepta']['refused'], [], true));
        $data['URLNotify']  = $this->request->getCurrentRequest()->getUriForPath($this->router->generate($this->paths['axepta']['response_at'], [], true));



        $blowfishString = '';

        foreach ($data as $key => $value) {
            if (array_key_exists($key, $data) && !empty($value)) {
                $blowfishString .= $key.'='.$value.'&';
            }
        }

        $blowfishString = rtrim($blowfishString, '&');
        $data['Data']   = bin2hex($this->encrypt($blowfishString, $blowfish));
        $data['Len']    = strlen($data['Data']);
        $message        = sprintf("*".$data['TransID']."*".$data['MerchantID']."*".$data['Amount']."*".$data['Currency']);

        $data['Mac'] = hash_hmac('sha256', $message, $hmac);
        $data['URL'] = static::URL;

        return $data;
    }

    public function valid($response)
    {
        $paymentMethod = $this->paymentMethodRepository->findOneBy(['method' => static::METHOD]);
        $config        = $this->serializer->denormalize($paymentMethod->getConfiguration(), Axepta::class);
        $data = [];

        $hmac          = $config->gethmacKey();
        $blowfish      = $config->getBlowfish();

        if ($response) {
            $response_decrypt = explode('&',str_replace("=","&",$this->decrypt(hex2bin(strtolower($response['Data'])),$blowfish)));
            $data_response = [];
            foreach( $response_decrypt as $k => $v )  ( $k & 1 ) === 0  ?  $data[$v] = null  : $data_response[$response_decrypt[$k-1]] = $v ;

            $data_response['MAC'] = strtolower(str_replace(' ', '', $data_response['MAC']));
            $hmacCore = sprintf($data_response['PayID']."*".$data_response['TransID']."*".$data_response['mid']."*".$data_response['Status']."*".$data_response['Code']);
            $mac = hash_hmac('sha256', $hmacCore, $hmac);
            $data = $data_response;
            if($data_response['MAC'] == $mac) {
                if(in_array($data_response['Status'],["OK","AUTHORIZED"]) && $data_response['Code'] == 00000000){
                    $data["Valid"] = true;
                    return $data;
                }
            }
        }
        $data["Valid"] = false;
        return $data;
    }

    private function encrypt($data, $key)
    {
        $l = strlen($key);

        if ($l < 16) {
            $key = str_repeat($key, ceil(16 / $l));
        }

        if ($m = strlen($data) % 8) {
            $data .= str_repeat("\x00", 8 - $m);
        }
        if (function_exists('mcrypt_encrypt')) {
            $val = mcrypt_encrypt(MCRYPT_BLOWFISH, $key, $data, MCRYPT_MODE_ECB);
        } else {
            $val = openssl_encrypt($data, 'BF-ECB', $key, OPENSSL_RAW_DATA | OPENSSL_NO_PADDING);
        }

        return $val;
    }

    private function decrypt($data, $key)
    {
        $l = strlen($key);
        if ($l < 16) {
            $key = str_repeat($key, ceil(16 / $l));
        }

        if (function_exists('mcrypt_encrypt')){
            $val = mcrypt_decrypt(MCRYPT_BLOWFISH, $key, $data, MCRYPT_MODE_ECB);
        } else {
            $val = openssl_decrypt($data, 'BF-ECB', $key, OPENSSL_RAW_DATA | OPENSSL_NO_PADDING);
        }

        return rtrim($val, "\0");
    }
}
