<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Shop\Model;

use CpCreation\VitiCore\Behavior\HasDisable;
use CpCreation\VitiCore\Behavior\Impl\Disable;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Ramsey\Uuid\Uuid;
use Ramsey\Uuid\UuidInterface;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * Class ShippingZone
 *
 * @ORM\Entity(repositoryClass="CpCreation\VitiCore\Shop\Repository\ShippingZoneRepository")
 * @ORM\Table(name="shop_shipping_zone")
 * @ORM\HasLifecycleCallbacks()
 */
class ShippingZone implements HasDisable
{
    use Disable;
    /**
     * @ORM\Id()
     * @ORM\Column(type="uuid")
     * @ORM\GeneratedValue(strategy="NONE")
     *
     * @var UuidInterface
     */
    private $id;

    /**
     * @ORM\Column(type="string")
     * @Assert\NotBlank()
     *
     * @var string
     */
    private $name;

    /**
     * @ORM\ManyToMany(targetEntity="ShippingCountry", inversedBy="zones")
     * @ORM\JoinTable(name="shop_shipping_countries_zones")
     * @ORM\OrderBy({"code" = "ASC"})
     * @Assert\Valid()
     *
     * @var Collection | ShippingCountry[]
     */
    private $countries;

    /**
     * @ORM\OneToMany(targetEntity="ShippingCounty", mappedBy="zone", cascade={"persist"})
     * @ORM\OrderBy({"code" = "ASC"})
     * @Assert\Valid()
     *
     * @var Collection | ShippingCounty[]
     */
    private $counties;


    public function __construct()
    {
        $this->id       = Uuid::uuid4();
        $this->countries = new ArrayCollection();
        $this->counties = new ArrayCollection();
        $this->disabled = false;
    }

    /**
     * @return UuidInterface
     */
    public function getId(): UuidInterface
    {
        return $this->id;
    }

    /**
     * @return string
     */
    public function getName(): ?string
    {
        return $this->name;
    }

    /**
     * @param string $name
     */
    public function setName(string $name): void
    {
        $this->name = $name;
    }

    /**
     * @param ShippingCountry $country
     * @return $this
     */
    public function addCountry(ShippingCountry $country)
    {
        if (!$this->countries->contains($country))
        {
            $country->addZone($this);
            $this->countries[] = $country;
        }

        return $this;
    }

    /**
     * @param ShippingCountry $country
     */
    public function removeCountry(ShippingCountry $country)
    {
        if ($this->countries->contains($country))
        {
            $this->countries->removeElement($country);
            $country->removeZone($this);
        }

    }

    /**
     * @return Collection | ShippingCountry[]
     */
    public function getCountries(): Collection
    {
        return $this->countries;
    }

    /**
     * @param ShippingCounty $county
     * @return $this
     */
    public function addCounty(ShippingCounty $county)
    {
        if (!$this->counties->contains($county))
        {
            $county->setZone($this);
            $this->counties[] = $county;
        }


        return $this;
    }

    /**
     * @param ShippingCounty $county
     */
    public function removeCounty(ShippingCounty $county)
    {
        if ($this->counties->contains($county))
        {
            $this->counties->removeElement($county);
            $county->setZone(null);
        }

    }

    /**
     * @return Collection | ShippingCounty[]
     */
    public function getCounties(): Collection
    {
        return $this->counties;
    }
}
