<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Cart\Model;

use CpCreation\VitiCore\Behavior\HasTimestamp;
use CpCreation\VitiCore\Behavior\Impl\Timestamp;
use CpCreation\VitiCore\Product\Model\Product;
use CpCreation\VitiCore\Product\Model\ProductDegressive;
use CpCreation\VitiCore\Product\Model\ProductGift;
use CpCreation\VitiCore\Product\Model\ProductVariant;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Ramsey\Uuid\Uuid;
use Ramsey\Uuid\UuidInterface;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * Class CartItem
 * @ORM\Entity(repositoryClass="CpCreation\VitiCore\Cart\Repository\CartItemRepository")
 * @ORM\Table(name="cart_cart_item")
 * @ORM\HasLifecycleCallbacks()
 */
class CartItem implements HasTimestamp
{
    use Timestamp;

    /**
     * @ORM\Id()
     * @ORM\Column(type="uuid")
     * @ORM\GeneratedValue(strategy="NONE")
     *
     * @var UuidInterface
     */
    private $id;

    /**
     * @ORM\Column(type="integer")
     * @Assert\NotBlank()
     * @var integer
     */
    private $quantity = 1;

    /**
     * @var Cart
     *
     * @ORM\ManyToOne(targetEntity="Cart", inversedBy="items")
     * @ORM\JoinColumn(name="cart_cart_id", referencedColumnName="id", onDelete="CASCADE")
     */
    private $cart;

    /**
     * @ORM\Column(type="integer", nullable=true)
     *
     * @var integer|null
     */
    private ?int $price = null;

    /**
     * @var ProductVariant
     *
     * @ORM\ManyToOne(targetEntity="CpCreation\VitiCore\Product\Model\ProductVariant")
     * @ORM\JoinColumn(name="product_variant_id", referencedColumnName="id")
     * @Assert\NotBlank()
     */
    private $variant;

    /**
     * @ORM\OneToMany(targetEntity="CpCreation\VitiCore\Product\Model\ProductGift",mappedBy="cartItem", cascade={"all"})
     *
     * @var Collection<ProductGift>
     */
    private $gifts;

    public function __construct(Product $product)
    {
        $this->id = Uuid::uuid4();
        $this->gifts = new ArrayCollection();
        if ($product->getVariantByDefault()) {
            $this->variant = $product->getVariantByDefault();
        }
    }

    /**
     * @return UuidInterface
     */
    public function getId(): UuidInterface
    {
        return $this->id;
    }

    /**
     * @return integer
     */
    public function getQuantity(): ?int
    {
        return $this->quantity;
    }

    /**
     * @param integer $quantity
     */
    public function setQuantity(int $quantity): void
    {
        $this->quantity = $quantity;
    }

    /**
     * @return ProductVariant
     */
    public function getVariant(): ?ProductVariant
    {
        return $this->variant;
    }

    /**
     * @param ProductVariant $variant
     */
    public function setVariant(ProductVariant $variant): void
    {
        $this->variant = $variant;
    }

    /**
     * @return Cart
     */
    public function getCart(): Cart
    {
        return $this->cart;
    }

    /**
     * @param Cart $cart
     */
    public function setCart(Cart $cart): void
    {
        $this->cart = $cart;
    }

    public function getPrice(): ?int
    {
        return $this->price;
    }

    public function setPrice(?int $price): void
    {
        $this->price = $price;
    }

    public function getTotal()
    {
        if ($this->price) {
            return $this->quantity * $this->getPrice();
        }
        return $this->getVariant()->getVariantPriceByGroupAvailable()->getPrice() * $this->quantity;
    }

    /**
     * @return ArrayCollection|Collection
     */
    public function getGifts(): ArrayCollection|Collection
    {
        return $this->gifts;
    }

    /**
     * @param ProductGift $gift
     * @return void
     */
    public function addGift(ProductGift $gift)
    {
        if ($this->gifts->contains($gift)) {
            return;
        }
        $gift->setCartItem($this);
        $this->gifts->add($gift);
    }

    /**
     * @param ProductGift $gift
     * @return void
     */
    public function removeGift(ProductGift $gift)
    {
        if (!$this->gifts->contains($gift)) {
            return;
        }
        $this->gifts->removeElement($gift);
    }

    public function removeGifts()
    {
        foreach ($this->getGifts() as $gift) {
            $this->gifts->removeElement($gift);
        }
    }

    public function getName(): string
    {
        return $this->getVariant()->getName();
    }


}
