<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Site;

use CpCreation\VitiCore\Bridge\Symfony\Bundle\Form\Site\ContactType;
use CpCreation\VitiCore\Bridge\Symfony\Bundle\Utils\ReCaptcha;
use CpCreation\VitiCore\Contact\Repository\ContactRepository;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use CpCreation\VitiCore\Depot\Repository\CategoryRepository;
use CpCreation\VitiCore\Event\Repository\EventRepository;
use Symfony\Component\Mailer\MailerInterface;
use Symfony\Component\Mime\Email;
use Symfony\Contracts\Translation\TranslatorInterface;

class ContactController extends AbstractController
{
    private $contactRepository;
    private $reCaptcha;
    private $translator;
    private $mailer;
    private $categoryRepository;
    private $eventRepository;

    public function __construct(
        ContactRepository $contactRepository,
        ReCaptcha $reCaptcha,
        TranslatorInterface $translator,
        MailerInterface $mailer,
        CategoryRepository $categoryRepository,
        EventRepository $eventRepository

    ) {
        $this->contactRepository    = $contactRepository;
        $this->reCaptcha            = $reCaptcha;
        $this->translator           = $translator;
        $this->mailer               = $mailer;
        $this->categoryRepository   = $categoryRepository;
        $this->eventRepository      = $eventRepository;
    }


    /**
     * @param Request $request
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse|\Symfony\Component\HttpFoundation\Response
     * @throws \Doctrine\ORM\NonUniqueResultException
     */
    public function indexAction(Request $request)
    {
        $categories = $this->categoryRepository->findCategories();
        $salons     = $this->eventRepository->findBy(['disabled' => false], ['beginAt' => 'ASC']);

        $contact            = $this->contactRepository->findOne();
        $recaptchaKeyPublic = false;
        if ($contact->isRecaptcha()) {
            $recaptchaKeyPublic = $contact->getRecaptchaKeyPublic();
        }
        $form = $this->createForm(ContactType::class);
        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $data = $form->getData();
            if ($contact->isRecaptcha()) {
                if (!$this->reCaptcha->verify($request->get('g-recaptcha-response'))) {
                    $this->addFlash('error', $this->translator->trans('contact.form.recaptcha.error'));

                    return $this->render(
                        'contact.html.twig',
                        [
                            'categories'         => $categories,
                            'salons'             => $salons,
                            'contact'            => $contact,
                            'classMenu'          => 'black',
                            'reCaptchaKeyPublic' => $recaptchaKeyPublic,
                            'form'               => $form->createView(),
                        ]
                    );
                }
            }

            $message = (new Email())
                ->to($contact->getEmailToMessage())
                ->from($contact->getEmailFrom(), $contact->getCompany())
                ->replyTo($data['email'], $data['lastname'])
                ->subject('Prise de contact depuis le site '.$contact->getCompany())
                ->text(
                    '
                    Nom : '.$data['lastname'].'<br/>
                    Téléphone : '.$data['phone'].'<br/>
                    E-mail : '.$data['email'].'<br/>
                    Message : '.$data['message'].'<br/>
                ', 'text/html'
                )
                ->addPart(
                    'Nom : '.$data['lastname'].' Téléphone : '.$data['phone'].' E-mail : '.$data['email'].' Message : '.$data['message'].$contact->getCompany(), 'text/plain'
                );

            if ($this->mailer->send($message)) {
                $this->addFlash('success', $this->translator->trans('contact.form.success'));

                return $this->redirectToRoute('contact');
            } else {
                $this->addFlash('error', $this->translator->trans('contact.form.error'));
            }

        }

        return $this->render(
            'contact.html.twig',
            [
                'categories'         => $categories,
                'salons'             => $salons,
                'contact'            => $contact,
                'classMenu'          => 'black',
                'reCaptchaKeyPublic' => $recaptchaKeyPublic,
                'form'               => $form->createView(),
            ]
        );
    }
}
