<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Admin;

use CpCreation\VitiCore\Popup\Model\Popup;
use CpCreation\VitiCore\Bridge\Symfony\Bundle\Form\Admin\PopupType;
use CpCreation\VitiCore\Popup\Repository\PopupRepository;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Security;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\Routing\Annotation\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;
use Symfony\Component\HttpFoundation\Request;

/**
 * @Security("is_granted('ROLE_ADMIN') or is_granted('ROLE_POPUP')")
 * Class PopupController
 * @package CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Admin
 * @Route("/popup")
 */
class PopupController extends AbstractController
{
    private $popupRepository;

    public function __construct(PopupRepository $popupRepository)
    {
        $this->popupRepository = $popupRepository;
    }

    /**
     * @Route("", name="admin_popup")
     */
    public function indexAction()
    {
        $popups = $this->popupRepository->findBy([], ['publishedAt' => 'DESC']);

        return $this->render('admin/popup/index.html.twig', [
            'items' => $popups,
        ]);
    }

    /**
     * @param Request $request
     * @Route("/new", name="admin_popup_new")
     */
    public function newAction(Request $request)
    {
        $form = $this->createForm(PopupType::class);
        $form->handleRequest($request);

        if($form->isSubmitted() && $form->isValid())
        {
            /** @var Popup $data */
            $data = $form->getData();
            $this->popupRepository->save($data);
            $this->addFlash('alert-success', 'Le pop-up '.$data->getTitle().' a été ajouté avec succès');

            return $this->redirectToRoute('admin_popup');
        }

        return $this->render('admin/popup/new.html.twig', [
            "form" => $form->createView(),
        ]);
    }

    /**
     * @param Popup $popup
     * @Route("/{id}/delete", name="admin_popup_delete")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function deleteAction(Popup $popup)
    {
        $this->popupRepository->delete($popup);
        $this->addFlash('alert-success', 'La récompense '.$popup->getTitle().' a été supprimé avec succès. ');

        return $this->redirectToRoute('admin_popup');
    }


    /**
     * @param Popup $popup
     * @param Request $request
     * @Route("/{id}/edit", name="admin_popup_edit")
     *
     * 
     */
    public function editAction(Popup $popup, Request $request)
    {
        $form = $this->createForm(PopupType::class, $popup);
        $form->handleRequest($request);
        if($form->isSubmitted() && $form->isValid())
        {
            /** @var Popup $data */
            $data = $form->getData();
            $data->setUpdatedAt(new \DateTime());
            $this->popupRepository->save($data);
            $this->addFlash('alert-success', 'Le pop-up '.$popup->getTitle().' a été modifié avec succès. ');

            return $this->redirectToRoute('admin_popup');
        }

        return $this->render('admin/popup/edit.html.twig', [
            "item" => $popup,
            "form" => $form->createView(),
        ]);
    }
}
