<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Admin;

use CpCreation\VitiCore\Bridge\Symfony\Bundle\Form\Admin\EventType;
use CpCreation\VitiCore\Event\Model\Event;
use CpCreation\VitiCore\Event\Repository\EventRepository;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Security;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\Routing\Annotation\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;
use Symfony\Component\HttpFoundation\Request;

/**
 * @Security("is_granted('ROLE_ADMIN') or is_granted('ROLE_EVENT')")
 * Class EventController
 * @package CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Admin
 * @Route("/event")
 */

class EventController extends AbstractController
{
    private $eventRepository;

    public function __construct(EventRepository $eventRepository)
    {
        $this->eventRepository = $eventRepository;
    }

    /**
     * @route("", name="admin_event")
     *
     * @return array
     */
    public function listAllEvents()
    {
        $event = $this->eventRepository->findBy([], ['beginAt' => 'ASC']);

        return $this->render('admin/event/index.html.twig', [
            "items" => $event,
        ]);
    }

    /**
     * @param Request $request
     * @Route("/new", name="admin_event_new")
     *
     * 
     */
    public function newAction(Request $request)
    {

        $form = $this->createForm(EventType::class);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            /** @var Event $data */
            $data = $form->getData();
            $this->eventRepository->save($data);
            $this->addFlash('alert-success', "L\'événement ".$data->getName().' a été ajouté avec succès');

            return $this->redirectToRoute('admin_event');
        }

        return $this->render('admin/event/new.html.twig', [
            "form" => $form->createView(),
        ]);
    }

    /**
     * @param Event $event
     * @param Request $request
     * @Route("/{id}/edit", name="admin_event_edit")
     *
     * 
     */
    public function editAction(Event $event, Request $request)
    {
        $form = $this->createForm(EventType::class, $event);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            /** @var Event $data */
            $data = $form->getData();
            $data->setUpdatedAt(new \DateTime());
            $this->eventRepository->save($data);
            $this->addFlash('alert-success', 'L\'événement '.$event->getName().' a été modifié avec succès');

            return $this->redirectToRoute('admin_event');
        }

        return $this->render('admin/event/edit.html.twig', [
            "item" => $event,
            "form" => $form->createView(),
        ]);
    }

    /**
     * @param Event $event
     * @Route("/{id}/delete", name="admin_event_delete")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function deleteAction(Event $event)
    {
        $this->eventRepository->delete($event);
        $this->addFlash('alert-success', 'L\'événement ' .$event->getName().' a été  supprimé avec succès.');

        return $this->redirectToRoute('admin_event');
    }

    /**
     * @param Event $event
     * @Route("/{id}/disable", name="admin_event_disable")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function disableAction(Event $event)
    {
        $event->setDisabled(1);
        $this->eventRepository->save($event);
        $this->addFlash('alert-success', 'L\'événement '.$event->getName().' a été dépublié.');

        return $this->redirectToRoute('admin_event');
    }

    /**
     * @param Event $event
     * @Route("/{id}/enable", name="admin_event_enable")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function enableAction(Event $event)
    {
        $event->setDisabled(0);
        $this->eventRepository->save($event);
        $this->addFlash('alert-success', 'L\'événement '.$event->getName().' a été publié.');

        return $this->redirectToRoute('admin_event');
    }

}
