<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Admin;

use CpCreation\VitiCore\Bridge\Symfony\Bundle\Form\Admin\DepotType;
use CpCreation\VitiCore\Depot\Model\Depot;
use CpCreation\VitiCore\Depot\Repository\DepotRepository;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Security;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\Routing\Annotation\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;
use Symfony\Component\HttpFoundation\Request;

/**
 * @Security("is_granted('ROLE_ADMIN') or is_granted('ROLE_DEPOT')")
 * Class DepotController
 * @package CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Admin
 * @Route("/depots")
 */
class DepotController extends AbstractController
{
    private $depotRepository;

    public function __construct(DepotRepository $depotRepository)
    {
        $this->depotRepository = $depotRepository;
    }

    /**
     * @route("", name="admin_depot")
     */
    public function listAllDepots()
    {

        $depots = $this->depotRepository->findAllOrderByCountryOrderByCategoryOrderPositionAdmin();

        return $this->render('admin/depot/index.html.twig', [
            "items" => $depots,
        ]);
    }

    /**
     * @param Request $request
     * @Route("/new", name="admin_depot_new")
     *
     * 
     */
    public function newAction(Request $request)
    {

        $form = $this->createForm(DepotType::class);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            /** @var Depot $data */
            $data = $form->getData();
            $this->depotRepository->save($data);
            $this->addFlash('alert-success', 'Le dépôt '.$data->getName().' a été ajouté avec succès');

            return $this->redirectToRoute('admin_depot');
        }

        return $this->render('admin/depot/new.html.twig', [
            "form" => $form->createView(),
        ]);
    }

    /**
     * @param Depot   $depot
     * @param Request $request
     * @Route("/{id}/edit", name="admin_depot_edit")
     *
     * 
     */
    public function editAction(Depot $depot, Request $request)
    {
        $form = $this->createForm(DepotType::class, $depot);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            /** @var Depot $data */
            $data = $form->getData();
            $this->depotRepository->save($data);
            $this->addFlash('alert-success', 'Le dépôt '.$depot->getName().' a été modifié avec succès');

            return $this->redirectToRoute('admin_depot');
        }

        return $this->render('admin/depot/edit.html.twig', [
            "item" => $depot,
            "form" => $form->createView(),
        ]);
    }

    /**
     * @param Depot $depot
     * @Route("/{id}/delete", name="admin_depot_delete")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function deleteAction(Depot $depot)
    {
        $this->depotRepository->delete($depot);
        $this->addFlash('alert-success', 'Le dépôt '.$depot->getName().' a été supprimé avec succès.');

        return $this->redirectToRoute('admin_depot');
    }

    /**
     * @param Depot $depot
     * @Route("/{id}/disable", name="admin_depot_disable")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function disableAction(Depot $depot)
    {
        $depot->setDisabled(1);
        $this->depotRepository->save($depot);
        $this->addFlash('alert-success', 'Le dépôt '.$depot->getName().' a été dépublié.');

        return $this->redirectToRoute('admin_depot');
    }

    /**
     * @param Depot $depot
     * @Route("/{id}/enable", name="admin_depot_enable")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function enableAction(Depot $depot)
    {
        $depot->setDisabled(0);
        $this->depotRepository->save($depot);
        $this->addFlash('alert-success', 'Le dépôt '.$depot->getName().' a été publié.');

        return $this->redirectToRoute('admin_depot');
    }

    /**
     * @param Depot $depot
     * @Route("/{id}/moveUp", name="admin_depot_move_up")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function moveUpAction(Depot $depot)
    {
        $depot->setPosition($depot->getPosition() - 1);
        $this->depotRepository->save($depot);
        $this->addFlash('alert-success', 'Le dépôt '.$depot->getName().' a été remonté.');

        return $this->redirectToRoute('admin_depot');
    }

    /**
     * @param Depot $depot
     * @Route("/{id}/moveDown", name="admin_depot_move_down")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function moveDownAction(Depot $depot)
    {
        $depot->setPosition($depot->getPosition() + 1);
        $this->depotRepository->save($depot);
        $this->addFlash('alert-success', 'Le dépôt '.$depot->getName().' a été descendu.');

        return $this->redirectToRoute('admin_depot');
    }
}
