<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Admin;

use CpCreation\VitiCore\Bridge\Symfony\Bundle\Form\Admin\DepotCategoryType;
use CpCreation\VitiCore\Depot\Model\Category;
use CpCreation\VitiCore\Depot\Repository\CategoryRepository;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Security;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;
use Symfony\Component\HttpFoundation\Request;

/**
 * @Security("is_granted('ROLE_ADMIN') or is_granted('ROLE_DEPOT')")
 * Class DepotCategoryController
 * @package CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Admin
 * @Route("/categories_depots")
 */
class DepotCategoryController extends AbstractController
{
    private $categoryRepository;

    public function __construct(CategoryRepository $categoryRepository)
    {
        $this->categoryRepository = $categoryRepository;
    }

    /**
     * @route("", name="admin_depot_category")
     */
    public function listAllDepots(): Response
    {
        $categories = $this->categoryRepository->findByPositionOrderAdmin();

        return $this->render('admin/depot/category/index.html.twig', [
            "items" => $categories,
        ]);
    }

    /**
     * @param Request $request
     * @Route("/new", name="admin_depot_category_new")
     *
     * 
     */
    public function newAction(Request $request): Response|RedirectResponse
    {

        $form = $this->createForm(DepotCategoryType::class);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            /** @var Category $data */
            $data = $form->getData();
            $this->categoryRepository->save($data);
            $this->addFlash('alert-success', 'La catégorie '.$data->getName().' a été ajouté avec succès');

            return $this->redirectToRoute('admin_depot_category');
        }

        return $this->render('admin/depot/category/new.html.twig', [
            "form" => $form->createView(),
        ]);
    }

    /**
     * @param Category   $category
     * @param Request $request
     * @Route("/{id}/edit", name="admin_depot_category_edit")
     *
     * 
     */
    public function editAction(Category $category, Request $request): Response|RedirectResponse
    {
        $form = $this->createForm(DepotCategoryType::class, $category);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            /** @var Category $data */
            $data = $form->getData();
            $this->categoryRepository->save($data);
            $this->addFlash('alert-success', 'La categorie '.$category->getName().' a été modifié avec succès');

            return $this->redirectToRoute('admin_depot_category');
        }

        return $this->render('admin/depot/category/edit.html.twig', [
            "item" => $category,
            "form" => $form->createView(),
        ]);
    }

    /**
     * @param Category $category
     * @Route("/{id}/delete", name="admin_depot_category_delete")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function deleteAction(Category $category)
    {
        $this->categoryRepository->delete($category);
        $this->addFlash('alert-success', 'La catégorie '.$category->getName().' a été supprimé avec succès.');

        return $this->redirectToRoute('admin_depot_category');
    }

    /**
     * @param Category $category
     * @Route("/{id}/moveUp", name="admin_depot_category_move_up")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function moveUpAction(Category $category)
    {
        $category->setPosition($category->getPosition() - 1);
        $this->categoryRepository->save($category);
        $this->addFlash('alert-success', 'La categorie '.$category->getName().' a été remonté.');

        return $this->redirectToRoute('admin_depot_category');
    }

    /**
     * @param Category $category
     * @Route("/{id}/moveDown", name="admin_depot_category_move_down")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function moveDownAction(Category $category)
    {
        $category->setPosition($category->getPosition() + 1);
        $this->categoryRepository->save($category);
        $this->addFlash('alert-success', 'La categorie '.$category->getName().' a été descendu.');

        return $this->redirectToRoute('admin_depot_category');
    }
}
