<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Admin;

use CpCreation\VitiCore\Bridge\Symfony\Bundle\Form\Admin\DeliveryMethodType;
use CpCreation\VitiCore\Shop\Model\DeliveryMethod;
use CpCreation\VitiCore\Shop\Model\DeliveryMethodCategory;
use CpCreation\VitiCore\Shop\Repository\DeliveryMethodRepository;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Security;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;

/**
 * @Security("is_granted('ROLE_ADMIN') or is_granted('ROLE_SHIPPING_METHOD')")
 * Class DeliveryMethodController
 * @package CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Admin
 * @Route("/delivery-method")
 */
class DeliveryMethodController extends AbstractController
{
    private $deliveryMethodRepository;

    public function __construct(DeliveryMethodRepository $deliveryMethodRepository)
    {
        $this->deliveryMethodRepository = $deliveryMethodRepository;
    }

    /**
     * @route("", name="admin_delivery_method")
     */
    public function indexAction(): Response
    {
        $methods = $this->deliveryMethodRepository->findByPositionOrderAdmin();

        return $this->render('admin/delivery_method/index.html.twig', [
            "items" => $methods,
        ]);

    }


    /**
     * @param Request $request
     * @Route("/new", name="admin_delivery_method_new")
     *
     * 
     */
    public function newAction(Request $request): Response|RedirectResponse
    {

        $form = $this->createForm(DeliveryMethodType::class);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            /** @var DeliveryMethodCategory $data */
            $data = $form->getData();
            $this->deliveryMethodRepository->save($data);
            $this->addFlash('alert-success', 'La catégorie a été ajouté avec succès');

            return $this->redirectToRoute('admin_delivery_method');

        }

        return $this->render('admin/delivery_method/new.html.twig', [
            "form" => $form->createView(),
        ]);
    }

    /**
     * @param DeliveryMethod $method
     * @param Request $request
     * @Route("/{id}/edit", name="admin_delivery_method_edit")
     *
     * 
     */
    public function editAction(DeliveryMethod $method, Request $request): Response|RedirectResponse
    {
        $form = $this->createForm(DeliveryMethodType::class, $method);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            /** @var DeliveryMethod $data */
            $data = $form->getData();
            $this->deliveryMethodRepository->save($data);
            $this->addFlash('alert-success', 'La méthode de livraison a été modifié avec succès');

            return $this->redirectToRoute('admin_delivery_method');

        }

        return $this->render('admin/delivery_method/edit.html.twig', [
            "item" => $method,
            "form" => $form->createView(),
        ]);
    }

    /**
     * @param DeliveryMethod $method
     * @Route("/{id}/delete", name="admin_delivery_method_delete")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function deleteAction(DeliveryMethod $method): RedirectResponse
    {
        $this->deliveryMethodRepository->softDelete($method);
        $this->addFlash('alert-success', 'La méthode de livraison a été supprimé avec succès.');

        return $this->redirectToRoute('admin_delivery_method');
    }

    /**
     * @param DeliveryMethod $method
     * @Route("/{id}/moveUp", name="admin_delivery_method_move_up")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function moveUpAction(DeliveryMethod $method):RedirectResponse
    {
        $method->setPosition($method->getPosition() - 1);
        $this->deliveryMethodRepository->save($method);
        $this->addFlash('alert-success', 'La méthode de livraison '.$method->getName().' a été remonté.');

        return $this->redirectToRoute('admin_delivery_method');
    }

    /**
     * @param DeliveryMethod $method
     * @Route("/{id}/moveDown", name="admin_delivery_method_move_down")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function moveDownAction(DeliveryMethod $method): RedirectResponse
    {
        $method->setPosition($method->getPosition() + 1);
        $this->deliveryMethodRepository->save($method);
        $this->addFlash('alert-success', 'La méthode de livraison '.$method->getName().' a été descendu.');

        return $this->redirectToRoute('admin_delivery_method');
    }

    /**
     * @param DeliveryMethod $method
     * @Route("/{id}/disable", name="admin_delivery_method_disable")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function disableAction(DeliveryMethod $method): RedirectResponse
    {
        $method->setDisabled(1);
        $this->deliveryMethodRepository->save($method);
        $this->addFlash('alert-success', 'La méthode de livraison '.$method->getName().' a été dépublié.');

        return $this->redirectToRoute('admin_delivery_method');
    }

    /**
     * @param DeliveryMethod $method
     * @Route("/{id}/enable", name="admin_delivery_method_enable")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function enableAction(DeliveryMethod $method): RedirectResponse
    {
        $method->setDisabled(0);
        $this->deliveryMethodRepository->save($method);
        $this->addFlash('alert-success', 'La méthode de livraison '.$method->getName().' a été publié.');

        return $this->redirectToRoute('admin_delivery_method');
    }
}
