<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Admin;

use CpCreation\VitiCore\Bridge\Symfony\Bundle\Form\Admin\DeliveryMethodCategoryType;
use CpCreation\VitiCore\Shop\Model\DeliveryMethodCategory;
use CpCreation\VitiCore\Shop\Repository\DeliveryMethodCategoryRepository;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Security;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;

/**
 * @Security("is_granted('ROLE_ADMIN')")
 *
 * Class DeliveryMethodCategoryController
 * @package CpCreation\VitiCore\Bridge\Symfony\Bundle\Controller\Admin
 * @Route("/delivery-method-category")
 */
class DeliveryMethodCategoryController extends AbstractController
{
    private $deliveryMethodCategoryRepository;

    public function __construct(DeliveryMethodCategoryRepository $deliveryMethodCategoryRepository)
    {
        $this->deliveryMethodCategoryRepository = $deliveryMethodCategoryRepository;
    }

    /**
     * @route("", name="admin_delivery_method_category")
     */
    public function indexAction(): Response
    {
        $categories = $this->deliveryMethodCategoryRepository->findAll();

        return $this->render('admin/delivery_method/category/index.html.twig', [
            "items" => $categories
        ]);
    }


    /**
     * @param Request $request
     * @Route("/new", name="admin_delivery_method_category_new")
     * 
     */
    public function newAction(Request $request): Response|RedirectResponse
    {

        $form = $this->createForm(DeliveryMethodCategoryType::class);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            /** @var DeliveryMethodCategory $data */
            $data = $form->getData();
            $this->deliveryMethodCategoryRepository->save($data);
            $this->addFlash('alert-success', 'La catégorie a été ajouté avec succès');

            return $this->redirectToRoute('admin_delivery_method_category');

        }

        return $this->render('admin/delivery_method/category/new.html.twig', [
            "form" => $form->createView(),
        ]);
    }

    /**
     * @param DeliveryMethodCategory $category
     * @param Request $request
     * @Route("/{id}/edit", name="admin_delivery_method_category_edit")
     *
     * 
     */
    public function editAction(DeliveryMethodCategory $category, Request $request): Response|RedirectResponse
    {
        $form = $this->createForm(DeliveryMethodCategoryType::class, $category);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            /** @var DeliveryMethodCategory $data */
            $data = $form->getData();
            $this->deliveryMethodCategoryRepository->save($data);
            $this->addFlash('alert-success', 'La catégorie a été modifié avec succès');

            return $this->redirectToRoute('admin_delivery_method_category');

        }

        return $this->render('admin/delivery_method/category/edit.html.twig', [
            "item" => $category,
            "form" => $form->createView(),
        ]);
    }

    /**
     * @param DeliveryMethodCategory $category
     * @Route("/{id}/delete", name="admin_delivery_method_category_delete")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function deleteAction(DeliveryMethodCategory $category):RedirectResponse
    {
        $this->deliveryMethodCategoryRepository->delete($category);
        $this->addFlash('alert-success', 'La catégorie a été supprimé avec succès.');

        return $this->redirectToRoute('admin_delivery_method_category');
    }

    /**
     * @param DeliveryMethodCategory $category
     * @Route("/{id}/moveUp", name="admin_delivery_method_category_move_up")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function moveUpAction(DeliveryMethodCategory $category):RedirectResponse
    {
        $category->setPosition($category->getPosition() - 1);
        $this->deliveryMethodCategoryRepository->save($category);
        $this->addFlash('alert-success', 'La catégorie '.$category->getName().' a été remonté.');

        return $this->redirectToRoute('admin_delivery_method_category');
    }

    /**
     * @param DeliveryMethodCategory $category
     * @Route("/{id}/moveDown", name="admin_delivery_method_category_move_down")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function moveDownAction(DeliveryMethodCategory $category):RedirectResponse
    {
        $category->setPosition($category->getPosition() + 1);
        $this->deliveryMethodCategoryRepository->save($category);
        $this->addFlash('alert-success', 'La catégorie '.$category->getName().' a été descendu.');

        return $this->redirectToRoute('admin_delivery_method_category');
    }

    /**
     * @param DeliveryMethodCategory $category
     * @Route("/{id}/disable", name="admin_delivery_method_category_disable")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function disableAction(DeliveryMethodCategory $category):RedirectResponse
    {
        $category->setDisabled(1);
        $this->deliveryMethodCategoryRepository->save($category);
        $this->addFlash('alert-success', 'La catégorie '.$category->getName().' a été dépublié.');

        return $this->redirectToRoute('admin_delivery_method_category');
    }

    /**
     * @param DeliveryMethodCategory $category
     * @Route("/{id}/enable", name="admin_delivery_method_category_enable")
     *
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function enableAction(DeliveryMethodCategory $category): RedirectResponse
    {
        $category->setDisabled(0);
        $this->deliveryMethodCategoryRepository->save($category);
        $this->addFlash('alert-success', 'La catégorie '.$category->getName().' a été publié.');

        return $this->redirectToRoute('admin_delivery_method_category');
    }
}
