<?php
/**
 * @author Colinet Julien
 */

namespace CpCreation\SkeletonBundle\EventListener\Doctrine;

use Doctrine\Common\EventSubscriber;
use Doctrine\ORM\Event\LifecycleEventArgs;
use Doctrine\ORM\Events;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

class LifecycleSubscriber implements EventSubscriber
{
    /**
     * @var EventDispatcherInterface
     */
    private $eventDispatcher;

    public function __construct(EventDispatcherInterface $eventDispatcher)
    {
        $this->eventDispatcher = $eventDispatcher;
    }

    public function dispatchEventIfClassExists(?string $eventClass, $model): void
    {
        if (!class_exists($eventClass)) {
            return;
        }

        $this->eventDispatcher->dispatch($eventClass, new $eventClass($model));
    }

    public function extractEventClass($model, string $verb): ?string
    {
        $entityClass = get_class($model);
        $matches     = null;
        preg_match('#App\\\(?<context>\w+)\\\Model\\\(?<model>\w+)#i', $entityClass, $matches);

        if (!(isset($matches['context']) && isset($matches['model']))) {
            return null;
        }

        return "App\\${matches['context']}\\Event\\${matches['model']}${verb}";
    }

    public function prePersist(LifecycleEventArgs $args): void
    {
        $model = $args->getEntity();
        $this->dispatchEventIfClassExists($this->extractEventClass($model, 'Create'), $model);
    }

    public function postPersist(LifecycleEventArgs $args): void
    {
        $model = $args->getEntity();
        $this->dispatchEventIfClassExists($this->extractEventClass($model, 'Created'), $model);
    }

    public function preUpdate(LifecycleEventArgs $args): void
    {
        $model = $args->getEntity();
        $this->dispatchEventIfClassExists($this->extractEventClass($model, 'Update'), $model);
    }

    public function postUpdate(LifecycleEventArgs $args): void
    {
        $model = $args->getEntity();
        $this->dispatchEventIfClassExists($this->extractEventClass($model, 'Updated'), $model);
    }

    public function preRemove(LifecycleEventArgs $args): void
    {
        $model = $args->getEntity();
        $this->dispatchEventIfClassExists($this->extractEventClass($model, 'Remove'), $model);
    }

    public function postRemove(LifecycleEventArgs $args): void
    {
        $model = $args->getEntity();
        $this->dispatchEventIfClassExists($this->extractEventClass($model, 'Removed'), $model);
    }

    public function getSubscribedEvents()
    {
        return [
            Events::prePersist,
            Events::preUpdate,
            Events::preRemove,
            Events::postPersist,
            Events::postPersist,
            Events::postRemove,
        ];
    }
}