<?php

/*
 * This file is part of the Serenity package.
 *
 * (c) CP Creation <web@cpcreation.fr>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Serenity\SaasBundle\EventListener;

use Serenity\CoreBundle\Contract\UserInterface;
use Serenity\SaasBundle\Doctrine\Repository\ProjectRepository;
use Serenity\SaasBundle\Exception\MissingHeaderException;
use Symfony\Bundle\SecurityBundle\Security;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\EventDispatcher\Attribute\AsEventListener;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\Routing\RouterInterface;
use Symfony\Component\Security\Http\Event\LoginSuccessEvent;

#[AsEventListener(event: LoginSuccessEvent::class, method: 'onLogin')]
final readonly class LoginSuccessListener
{
    public function __construct(private ProjectRepository $projectRepository, private Security $security,
        private RouterInterface $router,
        #[Autowire(env: 'SERENITY_SAAS')]
        private bool $serenitySaas)
    {
    }

    public function onLogin(LoginSuccessEvent $event): void
    {
        $firewallName = $event->getFirewallName();
        if (!$this->serenitySaas) {
            return;
        }

        if ('admin' === $firewallName && !$this->security->isGranted('ROLE_SUPER_ADMIN')) {
            $token = $event->getAuthenticatedToken();
            $user = $token->getUser();
            if (!$user instanceof UserInterface) {
                return;
            }
            $projects = $this->projectRepository->getProjectsByTenant($user);
            if (1 === \count($projects)) {
                $event->getRequest()->getSession()->set('project_identifier', $projects[0]->getIdentifier());

                return;
            }
            if ($this->security->getUser() && $this->security->isGranted('ROLE_TENANT')) {
                $response = new RedirectResponse($this->router->generate('admin_saas_select_project'));
                $event->setResponse($response);
            }
        }

        if ('app' === $firewallName) {
            if (!$event->getRequest()->headers->has('x-serenity-id')) {
                throw new MissingHeaderException('x-serenity-id');
            }
            $event->getRequest()->getSession()->set('project_identifier', $event->getRequest()->headers->get('x-serenity-id'));
        }
    }
}
