<?php

declare(strict_types=1);

/*
 * This file is part of the Serenity package.
 *
 * (c) CP Creation <web@cpcreation.fr>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Serenity\SaasBundle\Doctrine\Repository;

use Doctrine\ORM\QueryBuilder;
use Doctrine\Persistence\ManagerRegistry;
use Knp\Component\Pager\Pagination\PaginationInterface;
use Serenity\CoreBundle\Contract\SearchInterface;
use Serenity\CoreBundle\Doctrine\Contract\CrudRepositoryInterface;
use Serenity\CoreBundle\Doctrine\Repository\BaseRepository;
use Serenity\SaasBundle\DTO\TenantSearchDTO;
use Serenity\SaasBundle\Entity\Tenant;

final class TenantRepository extends BaseRepository implements CrudRepositoryInterface
{
    private const string ENTITY_CLASS = Tenant::class;
    private const string ALIAS = 'tn';

    public function __construct(ManagerRegistry $registry)
    {
        parent::__construct($registry, self::ENTITY_CLASS);
    }

    public function findOne(string $identifier): ?Tenant
    {
        $qb = $this->createQueryBuilder(self::ALIAS)
          ->andWhere(self::ALIAS.'.email.value = :email')
          ->setMaxResults(1)
          ->setParameter('email', $identifier);

        return $qb->getQuery()->getOneOrNullResult();
    }

    public function findAllPagination(int $page, int $limit = 30, ?QueryBuilder $queryBuilder = null): PaginationInterface
    {
        if (!$queryBuilder) {
            $queryBuilder = $this
              ->createQueryBuilder('t')
              ->addOrderBy('t.createdAt', 'DESC');
        }

        return $this->pagination($queryBuilder, $page, $limit);
    }

    /**
     * @param array<string, mixed> $searchBy
     */
    public function searchQuery(?SearchInterface $search = null, array $searchBy = []): ?QueryBuilder
    {
        if (null === $search && empty($searchBy)) {
            return null;
        }

        $qb = $this
            ->createQueryBuilder('t');

        if ($search && $search->getQuery()) {
            $qb->leftJoin('t.profile', 'profile')
              ->orWhere('LOWER(profile.lastname) LIKE :query')
              ->orWhere('LOWER(profile.firstname) LIKE :query')
              ->orWhere('LOWER(profile.company) LIKE :query')
              ->orWhere('LOWER(t.email.value) LIKE :query')
              ->setParameter('query', '%'.mb_strtolower($search->getQuery()).'%');
        }

        if ($search && $search->getSorter()) {
            $qb = $this->generateSorterQueryBuilder(TenantSearchDTO::SORTER, $search, $qb);
        }

        return $qb;
    }
}
